#pragma noiv               // Do not generate interrupt vectors
//-----------------------------------------------------------------------------
//   File:      periph.c
//   Contents:   Hooks required to implement USB peripheral function.
//
//   Copyright (c) 1997 AnchorChips, Inc. All rights reserved
//-----------------------------------------------------------------------------
// Modified for N2PK VNA
// Modifications (C) 2004 Dave Roberts G8KBB
// 14 Sept 2006 - extended for VNAccess.dll
// 08 Nov 2006 - significant rewrite
// 01/08/09 - rewrite for AD9912 VNA

#define ALLOCATE_EXTERN
#include "fx2.h"
#include "fx2regs.h"
#include "msa-usb.h"


//-----------------------------------------------------------------------------
// These are used to include various time / space tradeoffs to improve performance
//-----------------------------------------------------------------------------

// ADC resuts are stored in this buffer pending return to the PC
xdata at 0xE000 volatile unsigned long nResultMag = 0;
xdata at 0xE004 volatile unsigned long nResultPhase = 0;

//-----------------------------------------------------------------------------
// flags and data storage defined in fw.c
//-----------------------------------------------------------------------------
extern BOOL   GotSUD;         // Received setup data flag
extern BOOL   Sleep;
extern BOOL   Rwuen;
extern BOOL   Selfpwr;

//------------------------------------------------------------------------
// flags and variables defined here
//-----------------------------------------------------------------------------
BYTE   Configuration;      // Current configuration
BYTE   AlternateSetting;   // Alternate settings
BYTE   AdcReadsDone;       // number of ADC reads performed
BYTE   AdcTimeStart;       // timeout timer for ADC reset
BYTE   AdcReadDelay;       // delay counter (msec) before first ADC read
BYTE   ReturnStatus;       // status byte returned to controller
BYTE   LastCommand;        // what command did I last receive?
bit    PauseDataIn;        // Pause sending Data In frames during ADC conversion
bit    PauseDataOut;       // Pause processing Data Out frames during ADC conversion
bit bMagReady;
bit	bPhaseReady;
unsigned long nResultPhaseTempLong;
unsigned long nResultMagTempLong;
unsigned short nResultPhaseTempShort;
unsigned short nResultMagTempShort;


//-----------------------------------------------------------------------------
// function prototypes
//-----------------------------------------------------------------------------

void handle_in(void);
void handle_out(void);
void DoAdcMagRead();
void DoAdcPhaseRead();
void DoAdcBothRead();


//-----------------------------------------------------------------------------
// Task Dispatcher hooks
//   The following hooks are called by the task dispatcher.
//
// TD_Init() is caalled once at FW startup
// TD_Poll() is callled continually by FW main loop
//-----------------------------------------------------------------------------

void TD_Init(void)             // Called once at startup
{
	// set the CPU clock to 12MHz & disable clock out
	// should already hve been though!
	//CPUCS = (CPUCS & ~bmCLKSPD & !bmCLKOE);
	// set the CPU clock to 48MHz
	CPUCS = ((CPUCS & ~bmCLKSPD) | bmCLKSPD1) ;

	// we are just using the default values, so yes this is not necessary...
	EP1OUTCFG = 0x20;             // to enable it, 0xA0;
	EP1INCFG = 0x20;              // to enable it, 0xA0;
	SYNCDELAY;                    // see TRM section 15.14
	EP2CFG = 0xA2;                // EP2 output, bulk, 512 byte double buffered
	SYNCDELAY;
	EP4CFG = 0x20;                // EP4 output, bulk, DISABLED (enable with 0xA0)
	SYNCDELAY;
	EP6CFG = 0xE2;                // EP6 input, bulk, 512 byte double buffered
	SYNCDELAY;
	EP8CFG = 0x60;                // EP8 input, bulk, DISABLED (enable with 0xE0)
	bMagReady = bPhaseReady = FALSE;

	// out endpoints do not come up armed
	// since the defaults are double buffered we must write dummy byte counts twice
//	SYNCDELAY;
	EP2BCL = 0x80;                // arm EP2OUT by writing byte count w/skip.
	SYNCDELAY;
	EP2BCL = 0x80;
//	SYNCDELAY;
//  EP4BCL = 0x80;                // arm EP4OUT by writing byte count w/skip.
//  SYNCDELAY;
//  EP4BCL = 0x80;

	// enable dual autopointer feature
	AUTOPTRSETUP = 7; // |= 0x01;

	Rwuen = TRUE;                 // Enable remote-wakeup

	// clear main counters and flags. Enable IN and OUT handling
	AdcReadsDone = 0;
	ReturnStatus = 0;
	PauseDataIn = FALSE;
	PauseDataOut = FALSE;
	// configure timer 1 for usec delay timer function
	// 13 bit upcounter, clocked at 48MHz/12 == 4 MHz (250nsec)
	// so writing count to high byte of 13 bit counter will
	// count in units of 32*.25 = 8 usec.
	TMOD = 0;

	// set up ports for VNA interface
	IOA = 0;
	IOB = 0;
	IOC = 0;
	IOD = 0;
	IOE = 0;
	OEA = ~(bmMagData | bmPhaseData);;
	OEB = 0xFF;
	OEC = 0xFF;
	OED = 0xFF;
	OEE = 0xFF;
}


// Poll loop is called repeatedly by task dispatcher.
// It handles, messages from the PC, ADC conversions and responses
// the pause data in and out flags are not used in this version at the moment

void TD_Poll(void)             // Called repeatedly while the device is idle
{
//	if( PauseDataOut == FALSE )
		handle_out();
//	if( PauseDataIn == FALSE )
		handle_in();
}


//-----------------------------------------------------------------------------
// main process to handle output frames from host to USB
//-----------------------------------------------------------------------------

void handle_out(void)
{ 
	BYTE i;
 	BYTE clock;
	WORD count;

	// check EP2 EMPTY(busy) bit in EP2468STAT (SFR), core set's this bit when FIFO is empty
	// indicates message waiting for VNA
	if(!(EP2468STAT & bmEP2EMPTY) )
	{
		// clear in data pause just in case
		PauseDataIn = FALSE;
		// find how big packet is
		count = (EP2BCH << 8) + EP2BCL;
		// process the command
		switch( EP2FIFOBUF[0] )
		{
			// rest all port lines low
			case CmdMsaPortReset:
				LastCommand = CmdMsaPortReset;
				IOA = 0;
				IOB = 0;
				IOC = 0;
				IOD = 0;
				IOE = 0;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;
			case CmdMsaPortRawWrite:
				if( count >= CmdMsaPortWriteRawMinSize )
				{
					LastCommand = CmdMsaPortRawWrite;
					if( EP2FIFOBUF[CmdMsaPortRawWriteFlags] & 1 )
						IOA = EP2FIFOBUF[CmdMsaPortRawWritePortA];
					if( EP2FIFOBUF[CmdMsaPortRawWriteFlags] & 2 )
						IOB = EP2FIFOBUF[CmdMsaPortRawWritePortB];
					if( EP2FIFOBUF[CmdMsaPortRawWriteFlags] & 4 )
						IOC = EP2FIFOBUF[CmdMsaPortRawWritePortC];
					if( EP2FIFOBUF[CmdMsaPortRawWriteFlags] & 8 )
						IOD = EP2FIFOBUF[CmdMsaPortRawWritePortD];
					if( EP2FIFOBUF[CmdMsaPortRawWriteFlags] & 0x10 )
						IOE = EP2FIFOBUF[CmdMsaPortRawWritePortE];
				}
				else
					LastCommand = 0xff;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;
			// the following 5 cases handle the writing of ports A-E
			// in each case the number of bytes given are written to the respective port
			// and if clock is not zero it is treated as a clock line and pulsed high
			// then low for each byte.
			case CmdMsaPortAWrite:
				if( count >= CmdMsaPortWriteMinSize )
				{
					i = EP2FIFOBUF[CmdMsaPortWriteByteCount];
					clock = EP2FIFOBUF[CmdMsaPortWriteClock];
					AUTOPTR1H = MSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					AUTOPTR1L = LSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					LastCommand = CmdMsaPortAWrite;
					if( clock )
					{
						do
						{
							IOA = EXTAUTODAT1;
							IOA |= clock;
						} while( --i );
						IOA &= ~clock;
					}
					else
						do
						{
							IOA = EXTAUTODAT1;
						} while( --i );
				}
				else
					LastCommand = 0xff;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;
			case CmdMsaPortBWrite:
				LastCommand = CmdMsaPortBWrite;
				if( count >= CmdMsaPortWriteMinSize )
				{
					i = EP2FIFOBUF[CmdMsaPortWriteByteCount];
					clock = EP2FIFOBUF[CmdMsaPortWriteClock];
					AUTOPTR1H = MSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					AUTOPTR1L = LSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					if( clock )
					{
						do
						{
							IOB = EXTAUTODAT1;
							IOB |= clock;
						} while( --i );
						IOB &= ~clock;
					}
					else
						do
						{
							IOB = EXTAUTODAT1;
						} while( --i );
				}
				else
					LastCommand = 0xff;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;
			case CmdMsaPortCWrite:
				LastCommand = CmdMsaPortCWrite;
				if( count >= CmdMsaPortWriteMinSize )
				{
					i = EP2FIFOBUF[CmdMsaPortWriteByteCount];
					clock = EP2FIFOBUF[CmdMsaPortWriteClock];
					AUTOPTR1H = MSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					AUTOPTR1L = LSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					if( clock )
					{
						do
						{
							IOC = EXTAUTODAT1;
							IOC |= clock;
						} while( --i );
						IOC &= ~clock;
					}
					else
						do
						{
							IOC = EXTAUTODAT1;
						} while( --i );
				}
				else
					LastCommand = 0xff;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;
			case CmdMsaPortDWrite:
				LastCommand = CmdMsaPortDWrite;
				if( count >= CmdMsaPortWriteMinSize )
				{
					i = EP2FIFOBUF[CmdMsaPortWriteByteCount];
					clock = EP2FIFOBUF[CmdMsaPortWriteClock];
					AUTOPTR1H = MSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					AUTOPTR1L = LSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					if( clock )
					{
						do
						{
							IOD = EXTAUTODAT1;
							IOD |= clock;
						} while( --i );
						IOD &= ~clock;
					}
					else
						do
						{
							IOD = EXTAUTODAT1;
						} while( --i );
				}
				else
					LastCommand = 0xff;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;
			case CmdMsaPortEWrite:
				LastCommand = CmdMsaPortEWrite;
				if( count >= CmdMsaPortWriteMinSize )
				{
					i = EP2FIFOBUF[CmdMsaPortWriteByteCount];
					clock = EP2FIFOBUF[CmdMsaPortWriteClock];
					AUTOPTR1H = MSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					AUTOPTR1L = LSB( &EP2FIFOBUF[CmdMsaPortWriteByteOfs] );
					if( clock )
					{
						do
						{
							IOE = EXTAUTODAT1;
							IOE |= clock;
						} while( --i );
						IOE &= ~clock;
					}
					else
						do
						{
							IOE = EXTAUTODAT1;
						} while( --i );
				}
				else
					LastCommand = 0xff;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;

			// the following 3 cases are the main ADC input functions
			// each one peforms the requested ADC conversion and sets the
			// AdcReadsDone count
			case CmdMsaReadMagAdc:
				if( count >= CmdMsaReadMinSize )
				{
					DoAdcMagRead();
					bMagReady = TRUE;
					LastCommand = CmdMsaReadMagAdc;
					AdcReadsDone = 1;
					ReturnStatus |= AdcDataReady;
				}
				else
					LastCommand = 0xff;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;
			case CmdMsaReadPhaseAdc:
				if( count >= CmdMsaReadMinSize )
				{
					DoAdcPhaseRead();
					bPhaseReady = TRUE;
					LastCommand = CmdMsaReadPhaseAdc;
					AdcReadsDone = 1;
					ReturnStatus |= AdcDataReady;
				}
				else
					LastCommand = 0xff;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;
			case CmdMsaReadBothAdc:
				if( count >= CmdMsaReadMinSize )
				{
					DoAdcBothRead();
					bMagReady = TRUE;
					bPhaseReady = TRUE;
					LastCommand = CmdMsaReadBothAdc;
					AdcReadsDone = 2;
					ReturnStatus |= AdcDataReady;
				}
				else
					LastCommand = 0xff;
				EP2BCL = 0x80;          // re(arm) EP2OUT
				break;
			default:                  // Invalid request
				LastCommand = 0;
				EP2BCL = 0x80;          // re(arm) EP2OUT
//				EZUSB_STALL_EP0();      // Stall End Point 0
		}
	}
}

// there are three ADC read functions - one for each ADC and one for both
// structurally they are all the same
// Each will read N bits of ADC and average over nAverage goes
// clocking nBits of data which can be variable from 1 to 32
// the clock option if set is for the LT1865 as opposed to the AD7685 clocking
// clock is a programmable convert high delay time

void DoAdcBothRead()
{
	BYTE temp;
 	BYTE clock;
	BYTE nBits;
	BYTE nAverage = EP2FIFOBUF[CmdMsaReadAverage];
	if( !nAverage )
		nAverage = 1;
	nResultPhase = nResultMag = 0;
	do
	{
		temp = IOA | bmAdcConv;
		IOA = temp & ~bmAdcSerClk;
		clock = EP2FIFOBUF[CmdMsaReadClockDelay];
		nBits = EP2FIFOBUF[CmdMsaReadBits]-1;
		while( clock-- > 0 )
			;
		temp = IOA | bmAdcSerClk;
		IOA = temp & ~bmAdcConv;
		if( EP2FIFOBUF[CmdMsaReadClockOptions] & 1 )
			IOA &= ~bmAdcSerClk;
		if( nBits <= 16 )
		{
			nResultPhaseTempShort = nResultMagTempShort = 0;
			do
			{
				IOA |= bmAdcSerClk;
				nResultMagTempShort <<= 1;
				nResultPhaseTempShort <<= 1;
				temp = IOA;
				IOA &= ~bmAdcSerClk;
				if( (temp & bmMagData ) == 0 )
					nResultMagTempShort |= 1;
				if( (temp & bmPhaseData ) == 0 )
					nResultPhaseTempShort |= 1;
			} while ( nBits-- > 0 );
			nResultMag += nResultMagTempShort;
			nResultPhase += nResultPhaseTempShort;
		}
		else
		{
			nResultPhaseTempLong = nResultMagTempLong = 0;
			do
			{
				IOA |= bmAdcSerClk;
				nResultMagTempLong <<= 1;
				nResultPhaseTempLong <<= 1;
				temp = IOA;
				IOA &= ~bmAdcSerClk;
				if( (temp & bmMagData ) == 0 )
					nResultMagTempLong |= 1;
				if( (temp & bmPhaseData ) == 0 )
					nResultPhaseTempLong |= 1;
			} while ( nBits-- > 0 );
			nResultMag += nResultMagTempLong;
			nResultPhase += nResultPhaseTempLong;
		}
	} while ( --nAverage > 0 );
	nAverage = EP2FIFOBUF[CmdMsaReadAverage];
	if( !nAverage )
		nAverage = 1;
	if( nAverage != 1 )
	{
		nResultMag /= nAverage;
		nResultPhase /= nAverage;
	}
}

void DoAdcMagRead()
{
	BYTE temp;
 	BYTE clock;
	BYTE nBits;
	BYTE nAverage = EP2FIFOBUF[CmdMsaReadAverage];
	if( !nAverage )
		nAverage = 1;
	nResultPhase = nResultMag = 0;
	do
	{
		temp = IOA | bmAdcConv;
		IOA = temp & ~bmAdcSerClk;
		clock = EP2FIFOBUF[CmdMsaReadClockDelay];
		nBits = EP2FIFOBUF[CmdMsaReadBits]-1;
		while( clock-- > 0 )
			;
		temp = IOA | bmAdcSerClk;
		IOA = temp & ~bmAdcConv;
		if( EP2FIFOBUF[CmdMsaReadClockOptions] & 1 )
			IOA &= ~bmAdcSerClk;
		if( nBits <= 16 )
		{
			nResultMagTempShort = 0;
			do
			{
				IOA |= bmAdcSerClk;
				nResultMagTempShort <<= 1;
				temp = IOA;
				IOA &= ~bmAdcSerClk;
				if( (temp & bmMagData ) == 0 )
					nResultMagTempShort |= 1;
			} while ( nBits-- > 0 );
			nResultMag += nResultMagTempShort;
		}
		else
		{
			nResultMagTempLong = 0;
			do
			{
				IOA |= bmAdcSerClk;
				nResultMagTempLong <<= 1;
				temp = IOA;
				IOA &= ~bmAdcSerClk;
				if( (temp & bmMagData ) == 0 )
					nResultMagTempLong |= 1;
			} while ( nBits-- > 0 );
			nResultMag += nResultMagTempLong;
		}
	} while ( --nAverage > 0 );
	nAverage = EP2FIFOBUF[CmdMsaReadAverage];
	if( !nAverage )
		nAverage = 1;
	if( nAverage != 1 )
		nResultMag /= nAverage;
}

void DoAdcPhaseRead()
{
	BYTE temp;
 	BYTE clock;
	BYTE nBits;
	BYTE nAverage = EP2FIFOBUF[CmdMsaReadAverage];
	if( !nAverage )
		nAverage = 1;
	nResultPhase = 0;
	do
	{
		temp = IOA | bmAdcConv;
		IOA = temp & ~bmAdcSerClk;
		clock = EP2FIFOBUF[CmdMsaReadClockDelay];
		nBits = EP2FIFOBUF[CmdMsaReadBits]-1;
		while( clock-- > 0 )
			;
		temp = IOA | bmAdcSerClk;
		IOA = temp & ~bmAdcConv;
		if( EP2FIFOBUF[CmdMsaReadClockOptions] & 1 )
			IOA &= ~bmAdcSerClk;
		if( nBits <= 16 )
		{
			nResultPhaseTempShort = 0;
			do
			{
				IOA |= bmAdcSerClk;
				nResultPhaseTempShort <<= 1;
				temp = IOA;
				IOA &= ~bmAdcSerClk;
				if( (temp & bmPhaseData ) == 0 )
					nResultPhaseTempShort |= 1;
			} while ( nBits-- > 0 );
			nResultPhase += nResultPhaseTempShort;
		}
		else
		{
			nResultPhaseTempLong = 0;
			do
			{
				IOA |= bmAdcSerClk;
				nResultPhaseTempLong <<= 1;
				temp = IOA;
				IOA &= ~bmAdcSerClk;
				if( (temp & bmPhaseData ) == 0 )
					nResultPhaseTempLong |= 1;
			} while ( nBits-- > 0 );
			nResultPhase += nResultPhaseTempLong;
		}
	} while ( --nAverage > 0 );
	nAverage = EP2FIFOBUF[CmdMsaReadAverage];
	if( !nAverage )
		nAverage = 1;
	if( nAverage != 1 )
		nResultPhase /= nAverage;
}


//-----------------------------------------------------------------------------
// simple utility to wait until usec timer times out then stop it
//-----------------------------------------------------------------------------

//void usecDelayWait( void )
//{
//	if( TR0 )
//		while( TF0 == 0 )
//			;
//	TR0 = 0;
//	TF0 = 0;
//}



//-----------------------------------------------------------------------------
// This function handles all comms in to the PC
//-----------------------------------------------------------------------------

void handle_in(void)             // Called repeatedly while the device is idle
{
	//BYTE j;
	BYTE k;

	// check EP6 EMPTY  bit in EP2468STAT (SFR), core set's this bit when FIFO is empty
	// do this to avoid double buffering
	if((EP2468STAT & bmEP6EMPTY))
	// check EP6 FULL(busy) bit in EP2468STAT (SFR), core set's this bit when FIFO is full
	// do this for double buffering
	//if(!(EP2468STAT & bmEP6FULL))
	{
		AUTOPTRH2 = MSB( &EP6FIFOBUF );
		AUTOPTRL2 = LSB( &EP6FIFOBUF );

		// put last command into response
		EXTAUTODAT2 = LastCommand;
		// signal whether we thought VNA power was present
		//if( IOD & bmVNAPower )
		//	ReturnStatus &= ~VnaNoPower;
		//else
		//	ReturnStatus |= VnaNoPower;

		// put return status flags and current IO ports into reply 
		// plus number of ADC reads completed
		EXTAUTODAT2 = ReturnStatus;
		EXTAUTODAT2 = IOA;
		EXTAUTODAT2 = IOB;
		EXTAUTODAT2 = IOC;
		EXTAUTODAT2 = IOD;
		EXTAUTODAT2 = IOE;
		EXTAUTODAT2 = AdcReadsDone;
		// If a sequence of ADC reads is finished, copy output data to reply
		// then reset for next command
		k = 8;
		if( bMagReady )
		{
			AUTOPTR1H = MSB( &nResultMag );
			AUTOPTR1L = LSB( &nResultMag );
			EXTAUTODAT2 = EXTAUTODAT1; 
			EXTAUTODAT2 = EXTAUTODAT1; 
			EXTAUTODAT2 = EXTAUTODAT1; 
			EXTAUTODAT2 = EXTAUTODAT1; 
			k += 4;
			bMagReady = FALSE;
			PauseDataOut = FALSE;
			ReturnStatus &= ~( AdcDataReady | AdcNotResponding);
		}
		if( bPhaseReady )
		{
			AUTOPTR1H = MSB( &nResultPhase );
			AUTOPTR1L = LSB( &nResultPhase );
			EXTAUTODAT2 = EXTAUTODAT1; 
			EXTAUTODAT2 = EXTAUTODAT1; 
			EXTAUTODAT2 = EXTAUTODAT1; 
			EXTAUTODAT2 = EXTAUTODAT1; 
			k += 4;
			bPhaseReady = FALSE;
			PauseDataOut = FALSE;
			ReturnStatus &= ~( AdcDataReady | AdcNotResponding);
		}
		AdcReadsDone = 0;
		EP6BCH = 0;  
		SYNCDELAY;
		EP6BCL = k;
	}
}


//-----------------------------------------------------------------------------
// suspend and resume hooks - not used here
//-----------------------------------------------------------------------------

BOOL TD_Suspend(void)          // Called before the device goes into suspend mode
{
   return(TRUE);
}

BOOL TD_Resume(void)          // Called after the device resumes
{
   return(TRUE);
}


//-----------------------------------------------------------------------------
// Device Request hooks
//   The following hooks are called by the end point 0 device request parser.
//-----------------------------------------------------------------------------

BOOL DR_GetDescriptor(void)
{
   return(TRUE);
}

BOOL DR_SetConfiguration(void)   // Called when a Set Configuration command is received
{
   Configuration = SETUPDAT[2];
   return(TRUE);            // Handled by user code
}

BOOL DR_GetConfiguration(void)   // Called when a Get Configuration command is received
{
   EP0BUF[0] = Configuration;
   EP0BCH = 0;
   SYNCDELAY;               // check is this needed?
   EP0BCL = 1;
   return(TRUE);            // Handled by user code
}

BOOL DR_SetInterface(void)       // Called when a Set Interface command is received
{
   AlternateSetting = SETUPDAT[2];
   return(TRUE);            // Handled by user code
}

BOOL DR_GetInterface(void)       // Called when a Set Interface command is received
{
   EP0BUF[0] = AlternateSetting;
   EP0BCH = 0;
   SYNCDELAY;               // check is this needed?
   EP0BCL = 1;
   return(TRUE);            // Handled by user code
}

BOOL DR_GetStatus(void)
{
   return(TRUE);
}

BOOL DR_ClearFeature(void)
{
   return(TRUE);
}

BOOL DR_SetFeature(void)
{
   return(TRUE);
}

BOOL DR_VendorCmnd(void)
{
	return(TRUE);
}


//-----------------------------------------------------------------------------
// USB Interrupt Handlers
//   The following functions are called by the USB interrupt jump table.
//-----------------------------------------------------------------------------

// Setup Data Available Interrupt Handler
void ISR_Sudav(void) interrupt 
{
   GotSUD = TRUE;            // Set flag
   EZUSB_IRQ_CLEAR();
   USBIRQ = bmSUDAV;         // Clear SUDAV IRQ
}

// Setup Token Interrupt Handler
void ISR_Sutok(void) interrupt 
{
   EZUSB_IRQ_CLEAR();
   USBIRQ = bmSUTOK;         // Clear SUTOK IRQ
}

void ISR_Sof(void) interrupt 
{
   EZUSB_IRQ_CLEAR();
   USBIRQ = bmSOF;            // Clear SOF IRQ
}

void ISR_Ures(void) interrupt 
{
   // whenever we get a USB reset, we should revert to full speed mode
   pConfigDscr = pFullSpeedConfigDscr;
   ((CONFIGDSCR xdata *) pConfigDscr)->type = CONFIG_DSCR;
   pOtherConfigDscr = pHighSpeedConfigDscr;
   ((CONFIGDSCR xdata *) pOtherConfigDscr)->type = OTHERSPEED_DSCR;
   
   EZUSB_IRQ_CLEAR();
   USBIRQ = bmURES;         // Clear URES IRQ
}

void ISR_Susp(void) interrupt 
{
   Sleep = TRUE;
   EZUSB_IRQ_CLEAR();
   USBIRQ = bmSUSP;
}

void ISR_Highspeed(void) interrupt 
{
   if (EZUSB_HIGHSPEED())
   {
      pConfigDscr = pHighSpeedConfigDscr;
      ((CONFIGDSCR xdata *) pConfigDscr)->type = CONFIG_DSCR;
      pOtherConfigDscr = pFullSpeedConfigDscr;
      ((CONFIGDSCR xdata *) pOtherConfigDscr)->type = OTHERSPEED_DSCR;
   }

   EZUSB_IRQ_CLEAR();
   USBIRQ = bmHSGRANT;
}

#ifdef INCLUDE_OTHER_VECTORS

void ISR_Ep0ack(void) interrupt 0
{
}
void ISR_Stub(void) interrupt 0
{
}
void ISR_Ep0in(void) interrupt 0
{
}
void ISR_Ep0out(void) interrupt 0
{
}
void ISR_Ep1in(void) interrupt 0
{
}
void ISR_Ep1out(void) interrupt 0
{
}
void ISR_Ep2inout(void) interrupt 0
{
}
void ISR_Ep4inout(void) interrupt 0
{
}
void ISR_Ep6inout(void) interrupt 0
{
}
void ISR_Ep8inout(void) interrupt 0
{
}
void ISR_Ibn(void) interrupt 0
{
}
void ISR_Ep0pingnak(void) interrupt 0
{
}
void ISR_Ep1pingnak(void) interrupt 0
{
}
void ISR_Ep2pingnak(void) interrupt 0
{
}
void ISR_Ep4pingnak(void) interrupt 0
{
}
void ISR_Ep6pingnak(void) interrupt 0
{
}
void ISR_Ep8pingnak(void) interrupt 0
{
}
void ISR_Errorlimit(void) interrupt 0
{
}
void ISR_Ep2piderror(void) interrupt 0
{
}
void ISR_Ep4piderror(void) interrupt 0
{
}
void ISR_Ep6piderror(void) interrupt 0
{
}
void ISR_Ep8piderror(void) interrupt 0
{
}
void ISR_Ep2pflag(void) interrupt 0
{
}
void ISR_Ep4pflag(void) interrupt 0
{
}
void ISR_Ep6pflag(void) interrupt 0
{
}
void ISR_Ep8pflag(void) interrupt 0
{
}
void ISR_Ep2eflag(void) interrupt 0
{
}
void ISR_Ep4eflag(void) interrupt 0
{
}
void ISR_Ep6eflag(void) interrupt 0
{
}
void ISR_Ep8eflag(void) interrupt 0
{
}
void ISR_Ep2fflag(void) interrupt 0
{
}
void ISR_Ep4fflag(void) interrupt 0
{
}
void ISR_Ep6fflag(void) interrupt 0
{
}
void ISR_Ep8fflag(void) interrupt 0
{
}
void ISR_GpifComplete(void) interrupt 0
{
}
void ISR_GpifWaveform(void) interrupt 0
{
}
#endif



