
// The following ifdef block is the standard way of creating macros which make exporting 
// from a DLL simpler. All files within this DLL are compiled with the VNACCESS_EXPORTS
// symbol defined on the command line. this symbol should not be defined on any project
// that uses this DLL. This way any other project whose source files include this file see 
// VNACCESS_API functions as being imported from a DLL, wheras this DLL sees symbols
// defined with this macro as being exported.
#ifdef VNACCESS_EXPORTS
#define VNACCESS_API EXTERN_C __declspec(dllexport)
#else
#define VNACCESS_API EXTERN_C __declspec(dllimport)
#endif

//
// Latest DLL Version number...
//	2.00	10/13/2004
//	2.01	08/10/2005
//	2.02	08/11/2005
//
#define VNACCESS_VERSION	301

//
// Version Info.
// Return format is (Major * 100 + Minor) - 102 means Version 1.02, e.g.
//
VNACCESS_API int _stdcall VNAccessVersion ();

//
// The microsecond TickCount functions.
//
//////
//
// Like ::GetTickCount (), but in units of microseconds rather than milliseconds.
//
VNACCESS_API unsigned int _stdcall VNAccessGetusTickCount ();
//
// Returns some indication of the actual resolution of the counter, in units of microseconds.
// Thus, if we only have access to a counter that bumps every 2 milliseconds, this would return 2000.
//
VNACCESS_API unsigned int _stdcall VNAccessGetusTickCountResolution ();

//
// Typedefs
//
typedef unsigned int VNAHANDLE;

//
// Return codes
//
#define VNACCESS_SUCCESS		(0)	// Operation succeeded
#define VNACCESS_INVALIDHANDLE		(~0)	// Invalid VNAHANDLE
#define VNACCESS_CONVERSIONINPROGRESS	(1)	// ADC busy converting - try again later
#define VNACCESS_VNANOTRESPONDING	(2)	// VNA not respondng - no power, not plugged in, etc.
#define VNACCESS_DDSPOWEREDDOWN		(3)	// ADC read successfully, but one or both DDS has been powered down
// not part of VNAccess - added for USB
#define VNACCESS_ADCTIMEOUT VNACCESS_VNANOTRESPONDING

// Open Error Codes
#define VNACCESS_NOMEM			(100)	// Couldn't allocate class instance during open
#define VNACCESS_COULDNOTRESERVEPORT	(101)	// Couldn't reserve requested parallel port during open
#define VNACCESS_COULDNOTINFERIOBASE	(102)	// Couldn't infer IO Base Address from port name during open
#define VNACCESS_COULDNOTWRITEREADPORT	(103)	// Couldn't communicate with IO Port Address during open
#define VNACCESS_WRITEREADPORTEXCEPTION	(104)	// Attempts to communicate with IO Port generated an exception

// Map Error Codes
#define VNACCESS_ILLEGALMAP		(200)
// ADC Selection Error Codes
#define VNACCESS_ILLEGALADCFLAG		(300)


//
// Open/Close bookends.
//

//
// Opens a VNA.
//	pszPortToReserve	Parallel port ("LPT1", e.g.) to CreateFile(), thereby reserving it from other processes.
//	IOAddress		Hardware IO Address (0x378, e.g.) at which we can directly interact with the parallel port.
//	nOuts			Number of 'out' instructions to issue each time the parallel port is set.
//					0 = try to figure out the optimal number (will be one until machines get really fast).
//					1 = go as fast as you can
//					>1 = absolute number of 'out's per port write.
//	pdwErrorCode		On failed Open, this is filled in with extra error info.  Can be NULL.
//
// The VNAHANDLE returned is required by all other VNAccess functions.
// VNACCESS_INVALIDHANDLE is returned if VNAccessOpen fails.
//
VNACCESS_API VNAHANDLE _stdcall VNAccessOpen (LPCSTR pszPortToReserve, unsigned int IOAddress, unsigned int nOuts, unsigned int *pdwErrorCode);
//
// Close a VNA.
//	hVNA	a valid VNAHANDLE.
//
// Releases the parallel port, frees memory.
//
VNACCESS_API int _stdcall VNAccessClose (VNAHANDLE hVNA);

//
// Parallel port pin masks.
//
// Output port
#define VNACCESS_PPOM_D0	0x01	// DB25-2
#define VNACCESS_PPOM_D1	0x02	// DB25-3
#define VNACCESS_PPOM_D2	0x04	// DB25-4
#define VNACCESS_PPOM_D3	0x08	// DB25-5
#define VNACCESS_PPOM_D4	0x10	// DB25-6
#define VNACCESS_PPOM_D5	0x20	// DB25-7
#define VNACCESS_PPOM_D6	0x40	// DB25-8
#define VNACCESS_PPOM_D7	0x80	// DB25-9
// Status port
#define VNACCESS_PPSM_FAULT	0x08	// DB25-15
#define VNACCESS_PPSM_ONLINE	0x10	// DB25-13
#define VNACCESS_PPSM_PAPER	0x20	// DB25-12
#define VNACCESS_PPSM_nACK	0x40	// DB25-10
#define VNACCESS_PPSM_BUSY	0x80	// DB25-11, inverted (software correction applied)!
// Control port
#define VNACCESS_PPCM_STB	0x01	// DB25-1,  inverted (software correction applied)!
#define VNACCESS_PPCM_AUTOFD	0x02	// DB25-14, inverted (software correction applied)!
#define VNACCESS_PPCM_nINIT	0x04	// DB25-16
#define VNACCESS_PPCM_SELECT	0x08	// DB25-17, inverted (software correction applied)!
//
// The lowest level access functions.
//
// Values for dwFlags:
#define VNACCESS_MF_NOW		0x0001		// Make modification before returning
#define VNACCESS_MF_NEXT	0x0002		// Make modification at next naturally occuring out instruction
#define VNACCESS_MF_FQ_UD	0x0004		// Make modification next time FQ_UD is asserted

//
// Modifies the Parallel Port Output port (D0-D7).
//	hVNA		a valid VNAHANDLE
//	SetMask		Bits in the port to set (OR'ed VNACCESS_PPOM_D* values)
//	ClrMask		Bits in the port to clear (OR'ed VNACCESS_PPOM_D* values)
//	dwFlags		When the modification should occur (one of the VNACCESS_MF_* values).
//
// Returns:
//	Nothing of any use
//
VNACCESS_API int _stdcall VNAccessModifyOutPort (VNAHANDLE hVNA, WORD wSetMask, WORD wClrMask, unsigned int dwFlags);

//
// Reads the Parallel Port Status port.  Software adjusts for hardware inversions (SELECT).
//	hVNA		a valid VNAHANDLE
//
// Returns:
//	The value of the Status port.  Individual bits can be tested with the VNACCES_PPSM_* masks.
//
VNACCESS_API int _stdcall VNAccessReadStatusPort (VNAHANDLE hVNA);

//
// Modifies the Parallel Port Control port (STB,AUTOFD,nINIT,SELECT).  Software adjusts for hardware inversions.
//	hVNA		a valid VNAHANDLE
//	SetMask		Bits in the port to set (OR'ed VNACCESS_PPCM_* values)
//	ClrMask		Bits in the port to clear (OR'ed VNACCESS_PPCM_* values)
//	dwFlags		When the modification should occur (one of the VNACCESS_MF_* values).
//
// Returns:
//	Nothing of any use
//
VNACCESS_API int _stdcall VNAccessModifyCtrlPort (VNAHANDLE hVNA, WORD wSetMask, WORD wClrMask, unsigned int dwFlags);

//
// Set up the Activity-Control Line mappings.
//

//
// Maximum number of Parallel Port register bits.
//
#define VNACCESS_PP_MAXLINES	24
//
// Parallel port output pin identifiers.
//
#define VNACCESS_PP_D0		0	// DB25-2
#define VNACCESS_PP_D1		1	// DB25-3
#define VNACCESS_PP_D2		2	// DB25-4
#define VNACCESS_PP_D3		3	// DB25-5
#define VNACCESS_PP_D4		4	// DB25-6
#define VNACCESS_PP_D5		5	// DB25-7
#define VNACCESS_PP_D6		6	// DB25-8
#define VNACCESS_PP_D7		7	// DB25-9
#define VNACCESS_PP_STB		16	// DB25-1,  inverted (software correction applied)!
#define VNACCESS_PP_AUTOFD	17	// DB25-14, inverted (software correction applied)!
#define VNACCESS_PP_nINIT	18	// DB25-16
#define VNACCESS_PP_SELECT	19	// DB25-17, inverted (software correction applied)!
//
// Parallel port input pin identifiers.
//
#define VNACCESS_PP_FAULT	11	// DB25-15
#define VNACCESS_PP_ONLINE	12	// DB25-13
#define VNACCESS_PP_PAPER	13	// DB25-12
#define VNACCESS_PP_nACK	14	// DB25-10
#define VNACCESS_PP_BUSY	15	// DB25-11, inverted (software correction applied)!
//
// Logical Control Signal identifiers/masks.
//
#define VNACCESS_LCS_NOMODIFY	0x00000000	// Pseudo-mask, placeholder for unmodified lines.
#define VNACCESS_LCS_UNUSED	0x00000000	// Pseudo-mask, placeholder for unused inputs (or outputs, if you like!).
#define VNACCESS_LCS_ZERO	0x00000002	// Pseudo-signal: sets parallel port output low during activity
#define VNACCESS_LCS_ONE	0x00000004	// Pseudo-signal: sets parallel port output high during activity
// DDS logical control signals
#define VNACCESS_LCS_DDS_RFDATA		0x00000010	// RF DDS Data
#define VNACCESS_LCS_DDS_LODATA		0x00000020	// LO DDS Data
#define VNACCESS_LCS_DDS_W_Clk		0x00000040	// Common DDS W_CLK
#define VNACCESS_LCS_DDS_FQ_UD		0x00000080	// Common DDS FQ_UD
#define VNACCESS_LCS_DDS_RESET		0x00000100	// Common DDS Reset
// ADC logical control signals
#define VNACCESS_LCS_ADC_SCK_C		0x00001000	// Common SCK
#define VNACCESS_LCS_ADC_nCS_1		0x00002000	// nCS for first ADC
#define VNACCESS_LCS_ADC_nSDO_1		0x00004000	// nSDO for first ADC
#define VNACCESS_LCS_ADC_SDI_C		0x00008000	// Common SDO (for the LTC2440)
#define VNACCESS_LCS_ADC_nCS_2		0x00010000	// nCS for second ADC
#define VNACCESS_LCS_ADC_nSDO_2		0x00020000	// nSDO for second ADC
#define VNACCESS_LCS_ADC_OZ_LINEFEED	0x00040000	// For OZ LTC2410 external oscillator speed selection
#define VNACCESS_LCS_ADC_OZ_SELECT	0x00080000	// For OZ LTC2410 external oscillator speed selection
// Attenuator logical control signals
#define VNACCESS_LCS_ATTEN0		0x00100000	// Atten0
#define VNACCESS_LCS_ATTEN1		0x00200000	// Atten1
#define VNACCESS_LCS_ATTEN2		0x00400000	// Atten2
// Test Set Sw logical control signal
#define VNACCESS_LCS_TEST_SET_SW	0x00800000	// Test Set Sw(2 rx)

typedef int VNACCESS_PinMapping_t[VNACCESS_PP_MAXLINES];
//
// Get/Set the Activity-Control Signal Mappings.
//	hVNA			a valid VNAHANDLE.
//	m			a physical pin to logical control signal mapping array.
//
// Can return:
//	VNACCESS_SUCCESS	mapping accepted
//	VNACCESS_INVALIDMAP	mapping not accepted.
//
// Each element of m can have at most one bit set.  The default maps are those specified for VNACCESS_PLCSM_0.
//
VNACCESS_API int _stdcall VNAccessSetDDSControlSignalMapping (VNAHANDLE hVNA, VNACCESS_PinMapping_t m);
VNACCESS_API int _stdcall VNAccessGetDDSControlSignalMapping (VNAHANDLE hVNA, VNACCESS_PinMapping_t m);
VNACCESS_API int _stdcall VNAccessSetADCControlSignalMapping (VNAHANDLE hVNA, VNACCESS_PinMapping_t m);
VNACCESS_API int _stdcall VNAccessGetADCControlSignalMapping (VNAHANDLE hVNA, VNACCESS_PinMapping_t m);
VNACCESS_API int _stdcall VNAccessSetAttenControlSignalMapping (VNAHANDLE hVNA, VNACCESS_PinMapping_t m);
VNACCESS_API int _stdcall VNAccessGetAttenControlSignalMapping (VNAHANDLE hVNA, VNACCESS_PinMapping_t m);
VNACCESS_API int _stdcall VNAccessSetTestSetSwControlSignalMapping (VNAHANDLE hVNA, VNACCESS_PinMapping_t m);
VNACCESS_API int _stdcall VNAccessGetTestSetSwControlSignalMapping (VNAHANDLE hVNA, VNACCESS_PinMapping_t m);
//
// Predefined Logical Control Signal mappings (unspecified pins are VNACCESS_LCS_UNUSED).
//	DDSMap:
//		VNACCESS_PLCSM_0, VNACCESS_PLCSM_1:
//			DDSMap[VNACCESS_PP_D0] = VNACCESS_LCS_DDS_RFDATA;
//			DDSMap[VNACCESS_PP_D1] = VNACCESS_LCS_DDS_LODATA;
//			DDSMap[VNACCESS_PP_D2] = VNACCESS_LCS_DDS_W_Clk;
//			DDSMap[VNACCESS_PP_D3] = VNACCESS_LCS_DDS_FQ_UD;
//			DDSMap[VNACCESS_PP_D4] = VNACCESS_LCS_DDS_RESET;
//	ADCMap:
//		VNACCESS_PLCSM_0:
//			ADCMap[VNACCESS_PP_D0] = VNACCESS_LCS_ADC_SDI_C;
//			ADCMap[VNACCESS_PP_D5] = VNACCESS_LCS_ADC_SCK_C;
//			ADCMap[VNACCESS_PP_D6] = VNACCESS_LCS_ADC_nCS_1;
//			ADCMap[VNACCESS_PP_BUSY] = VNACCESS_LCS_ADC_nSDO_1;
//			ADCMap[VNACCESS_PP_D7] = VNACCESS_LCS_ADC_nCS_2;
//			ADCMap[VNACCESS_PP_PAPER] = VNACCESS_LCS_ADC_nSDO_2;
//		VNACCESS_PLCSM_1:
//			ADCMap[VNACCESS_PP_D5] = VNACCESS_LCS_ADC_SCK_C;
//			ADCMap[VNACCESS_PP_D6] = VNACCESS_LCS_ADC_nCS_1;
//			ADCMap[VNACCESS_PP_BUSY] = VNACCESS_LCS_ADC_nSDO_1;
//			ADCMap[VNACCESS_PP_SELECT] = VNACCESS_LCS_ADC_OZ_SELECT;
//			ADCMap[VNACCESS_PP_AUTOFD] = VNACCESS_LCS_ADC_OZ_LINEFEED;
//	AttenMap:
//		VNACCESS_PLCSM_0:
//			AttenMap[VNACCESS_PP_AUTOFD] = VNACCESS_LCS_ATTEN0;
//			AttenMap[VNACCESS_PP_nINIT] = VNACCESS_LCS_ATTEN1;
//			AttenMap[VNACCESS_PP_SELECT] = VNACCESS_LCS_ATTEN2;
//	TestSetSwMap:
//		VNACCESS_PLCSM_0:
//			TestSetSwMap[VNACCESS_PP_STB] = VNACCESS_LCS_TEST_SET_SW;
//
#define VNACCESS_PLCSM_0	0	// N2PK Logical Control Signal Mapping
#define VNACCESS_PLCSM_1	1	// OZ Logical Control Signal Mapping
VNACCESS_API int _stdcall VNAccessSetPredefinedControlSignalMapping (VNAHANDLE hVNA, int PLCSMIndex);


//
// ADC Selection flags.
//
#define VNACCESS_ADC_DEFAULT	0	// Default ADC - Same behavior as VNACCESS_ADC_1, for now
#define VNACCESS_ADC_1		1	// First Detector - the standard N2PK VNA
#define VNACCESS_ADC_2		2	// Second Detector
#define VNACCESS_ADC_BOTH	3	// Both Detectors

//
// Check the VNA hardware/communications status.
//	hVNA			a valid VNAHANDLE.
//	usBusyWaitForEOC	Time to wait (in microseconds) for a conversion to finish (i.e. see a state change from the VNA).
//	ADCFlag			flag describing which ADC(s) to manipulate (from VNACCESS_ADC_*)
//
// Can return:
//	VNACCESS_SUCCESS		The VNA is responding and an ADC Result is ready.
//	VNACCESS_DDSPOWEREDDOWN		The VNA is responding, and ADC Result is ready, but the DDS chips are powered down.
//	VNACCESS_CONVERSIONINPROGRESS	The VNA is responding and an ADC conversion is in progress.
//	VNACCESS_VNANOTRESPONDING	The VNA is not responding.
//	VNACCESS_INVALIDHANDLE
//
VNACCESS_API int _stdcall VNAccessStatus (VNAHANDLE hVNA, unsigned int usBusyWaitForEOC, int ADCFlag);

//
// Get the computed nanoseconds-per-out metric.
//	hVNA			a valid VNAHANDLE.
//
// This will return (unsigned int)(~0) under the following conditions:
//	You did not specify 0 for nOuts in VNAccessOpen ().
//	The actual number of nanoseconds per out is >= 2^32-1.
//
// So, valid return values are 1 - (2^32-1).
//
VNACCESS_API unsigned int _stdcall VNAccessGetnsPerOut (VNAHANDLE hVNA);

//
// Power down the DDS chips if the ADC has not been read for a while.
//	hVNA			a valid VNAHANDLE.
//	msIdleTime		amount of idle time to wait, in milliseconds, before powering down the DDS chips.
//					0 => never automatically power down the DDS chips
//	CB			Pointer to Callback function called whenever the power down status may be changing (can be NULL)
//	ud			User-data that is supplied when CB is called (can be NULL)
//
// Returns:
//	VNACCESS_INVALIDHANDLE or the previous msIdleTime value
//
// This function lets you specify how long to wait before powering down the DDS chips.
// The idle timer is reset by calls to VNAccessIdleDDSPowerDown (), VNAccessSetDDS(), VNAccessDDSFQ_UD(),
//	VNAccessStartADC(), VNAccessGetADCResult(), VNAccessRaiseADCnCS().
//
VNACCESS_API unsigned int _stdcall VNAccessIdleDDSPowerDown (VNAHANDLE hVNA, unsigned int msIdleTime, void _stdcall CB(int, void *), void *ud);
//
// Retrieves the power down status of the DDS chips.
//	hVNA		a valid VNAHANDLE
//
// Returns:
//	non-zero iff one or both DDS chips are powered down.
//
VNACCESS_API int _stdcall VNAccessDDSPowerDown (VNAHANDLE hVNA);

//
// The obvious "do it" functions.
//

//
// Set the state of both (RF and LO) Direct Digital Synthesizers.
//	hVNA		a valid VNAHANDLE.
//	dw??TuningWord	32-bit DDS TuningWord
//	dw??PhaseOffset	5-bit DDS phase offset
//	b??PowerDown	Powers down the corresponding if non-zero
//	pusTickCount	Receives microsecond tick count of when FQ_UD went high. (== (~0) => don't apply FQ_UD)
//
// The State of both DDSs are set completely and simultaneously.
// A "synchronization reset" is performed (strobe reset, strobe FQ_UD) if in CFP mode, leaving the DDS chips in parallel load.
// This functions never returns VNACCESS_CONVERSIONINPROGRESS (setting the DDS state is independent of the ADC).
// By specifying VNACCESS_SETDDS_NO_FQ_UD for pusTickCount, you can clock new values into the DDS shift regsiters without
// transferring those values to the DDS cores - use VNAcessDDSFQ_UD() to complete the operation.
//
#define VNACCESS_SETDDS_NO_FQ_UD	((unsigned int *)(~0))
VNACCESS_API int _stdcall VNAccessSetDDS (VNAHANDLE hVNA,
				 unsigned int dwRFTuningWord, unsigned int dwRFPhaseOffset, BOOL bRFPowerDown,
				 unsigned int dwLOTuningWord, unsigned int dwLOPhaseOffset, BOOL bLOPowerDown,
				 unsigned int *pusTickCount);
//
// Pulses FQ_UD.
//	hVNA		a valid VNAHANDLE
//	pusTickCount	Receives microsecond tick count of when FQ_UD went high.
//
// Returns:
//	VNACCESS_SUCCESS.
//
// This function only makes sense after a call to VNAccessSetDDS() with a final parameter of VNACCESS_SETDDS_NO_FQ_UD.
//
VNACCESS_API int _stdcall VNAccessDDSFQ_UD (VNAHANDLE hVNA, unsigned int *pusTickCount);
//
// Resets both DDS chips.
//
// This should be done at power up, and any other time you wish to synchronize the chips.
//
VNACCESS_API int _stdcall VNAccessResetDDS (VNAHANDLE hVNA);
//
// Sets/Gets the CFP (Continuous Frequency Programming) mode.
//
// CFP mode "on" means synchronization is assumed, off means it happens each time the DDS state is set.
//
VNACCESS_API int _stdcall VNAccessSetDDSCFP (VNAHANDLE hVNA, BOOL bCFP);
VNACCESS_API int _stdcall VNAccessGetDDSCFP (VNAHANDLE hVNA, BOOL *pbCFP);
//
// Gets the synchronization state of the DDS chips.
//
VNACCESS_API int _stdcall VNAccessGetDDSSync (VNAHANDLE hVNA, BOOL *pbSync);


//
// Set/Get the Speed/Resolution of the LTC2440 ADC.
//	dwADCSpeed		speed specification (from VNACCESS_ADCSPEED_*)
//	pdwADCSpeed		pointer to unsigned int that receives current speed specification
//	ADCFlag			flag describing which ADC(s) to manipulate  (from VNACCESS_ADC_*)
//
// Speed Specifications (dwADCSpeed):
#define VNACCESS_ADCSPEED_3520	0x01
#define VNACCESS_ADCSPEED_1760	0x02
#define VNACCESS_ADCSPEED_880	0x03
#define VNACCESS_ADCSPEED_880a	0x00
#define VNACCESS_ADCSPEED_440	0x04
#define VNACCESS_ADCSPEED_220	0x05
#define VNACCESS_ADCSPEED_110	0x06
#define VNACCESS_ADCSPEED_55	0x07
#define VNACCESS_ADCSPEED_28	0x08
#define VNACCESS_ADCSPEED_14	0x09
#define VNACCESS_ADCSPEED_7	0x0F
//
#define VNACCESS_ADCSPEED_OZ_10MS	0x10F
#define VNACCESS_ADCSPEED_OZ_20MS	0x20F
#define VNACCESS_ADCSPEED_OZ_40MS	0x30F
#define VNACCESS_ADCSPEED_OZ_133MS	0x00F
VNACCESS_API int _stdcall VNAccessSetADCSpeed (VNAHANDLE hVNA, unsigned int dwADCSpeed, int ADCFlag);
VNACCESS_API int _stdcall VNAccessGetADCSpeed (VNAHANDLE hVNA, unsigned int *pdwADCSpeed, int ADCFlag);

//
// Start a new ADC conversion.
//	hVNA			a valid VNAHANDLE.
//	usTickCount		Tick Count timestamp for beginning of setup period.
//	usSetupMin		Number of micrseconds we must wait (relative to usTickCount) before starting conversion
//					 (0 means don't wait for this at all).
//	usBusyWaitForEOC	Number of microseconds we are allowed to busy wait for end of conversion.
//	pusTickCount		Receives microsecond tick count of when conversion starts
//	ADCFlag			flag describing which ADC(s) to manipulate (from VNACCESS_ADC_*)
//
// Conversions are started by reading the last ADC Result.
// This call is functionally identical to a VNAccessGetADCResult that tosses the result,
// but is slightly more efficient in that it does not need to read the complete result from the ADC.
// This function also allows some additional waiting for DDS-to-Detector setup time.
//
// This function can return VNACCESS_CONVERSIONINPROGRESS.
// This function can return VNACCESS_DDSPOWEREDDOWN which means a conversion result was returned,
//	but the DDS chips are powered down.
//
VNACCESS_API int _stdcall VNAccessStartADC (VNAHANDLE hVNA, unsigned int usTickCount, unsigned int usSetupMin, unsigned int usBusyWaitForEOC, unsigned int *pusTickCount, int ADCFlag);
//
// Read an ADC Result (and start a new conversion).
//	hVNA			a valid VNAHANDLE.
//	usBusyWaitForEOC	number of microseconds we are allowed to busy wait for end of conversion.
//	pdwADCResult		a pointer to a unsigned int that receives the ADC Result.
//	dwMinADCResultBits	the minimum number of (most significant) bits required in ADCResult
//	pdwActualADCResultBits	a pointer to a unsigned int that receives the actual number of valid (MS) bits in ADCResult
//	usTickCOunt		microsecond tick count timestamp for beginning of setup period.
//	usSetupMin		microseconds of setup time (relative to usTickCount) before next conversion can start
//	pusTickCount		a pointer to a unsigned int that receives microsecond tick count of when the (next) conversion starts
//					VNACCESS_NO_START means don't start next conversion.  We keep the next conversion
//					from starting by leaving nCS low and SCK high.  Start the next conversion from
//					this state by calling VNAccessRaiseADCnCS ().
//	ADCFlag			flag describing which ADC(s) to manipulate (from VNACCESS_ADC_*)
//	pdwADCResult2		a pointer to an unsigned int that receives the second detector ADC result -
//					ONLY when ADCFlag == VNACCESS_ADC_BOTH!
//
// This function can return VNACCESS_CONVERSIONINPROGRESS.
// This function can return VNACCESS_DDSPOWEREDDOWN which means a conversion was started,
//	but the DDS chips are powered down.
//
#define VNACCESS_NO_START	((unsigned int *)(~0))
VNACCESS_API int _stdcall VNAccessGetADCResult (VNAHANDLE hVNA,
				       unsigned int usBusyWaitForEOC, unsigned int *pdwADCResult,
				       unsigned int dwMinADCResultBits, unsigned int *pdwActualADCResultBits,
				       unsigned int usTickCount, unsigned int usSetupMin,
				       unsigned int *pusTickCount,
				       int ADCFlag, unsigned int *pdwADCResult2);
//
// Start ADC conversion after VNAccessGetADCResult (..., VNACCESS_NO_START, ...);
//	hVNA			a valid VNAHANDLE.
//	usTickCOunt		microsecond tick count timestamp for beginning of setup period.
//	usSetupMin		microseconds of setup time (relative to usTickCount) before next conversion can start
//	pusTickCount		a pointer to a unsigned int that receives microsecond tick count of when the (next) conversion starts
//	ADCFlag			flag describing which ADC(s) to manipulate (from VNACCESS_ADC_*)
//
VNACCESS_API int _stdcall VNAccessRaiseADCnCS (VNAHANDLE hVNA, unsigned int usTickCount, unsigned int usSetupMin, unsigned int *pusTickCount, int ADCFlag);

//
// The Atten Activity.
//
#define VNACCESS_ATTEN_0	0x0
#define VNACCESS_ATTEN_1	0x1
#define VNACCESS_ATTEN_2	0x2
#define VNACCESS_ATTEN_3	0x3
#define VNACCESS_ATTEN_4	0x4
#define VNACCESS_ATTEN_5	0x5
#define VNACCESS_ATTEN_6	0x6
#define VNACCESS_ATTEN_7	0x7
VNACCESS_API int _stdcall VNAccessSetAtten (VNAHANDLE hVNA, int Atten);


//
// The Test Set Sw Activity.
//
#define VNACCESS_TEST_SET_SW_0	0x0
#define VNACCESS_TEST_SET_SW_1	0x1
VNACCESS_API int _stdcall VNAccessSetTestSetSw (VNAHANDLE hVNA, int TestSetSw);

//
// Some convenient translation functions, based on the hardware data sheets.
//

//
// These functions provide conversion between frequencies and Tuning Words.
// Our convention is to express frequencies in Hz x 10, but, but the
// computation itself does not depend on this.
//
// So, to sythesize  3.5 MHz using 148.344 MHz Reference Clock, you could compute the tuning word with:
//	unsigned int TW = VNAccessFreqToTW (1483440000, 35000000);	// Note the "extra" zeros!
//
VNACCESS_API unsigned int _stdcall VNAccessFreqToTW (unsigned int RefClk, unsigned int F0);
VNACCESS_API unsigned int _stdcall VNAccessTWToFreq (unsigned int RefClk, unsigned int TW);

//
// This function converts an ADCResult to microvolts and indicates over-range/under-range.
//
// The notion of microvolts comes from three things:  1.  VRef+ of the ADC is 2.5v and VRef- is ground,
// 2. The ADC's notion of zero is (VRef+ - VRef-)/2, and ADCResults are 2's complement counts relative
// to that zero, and 3. historical VNA detectors have reported microvolts that are arithmetically
// opposite (multiplied by -1) from the results of this VNA's ADC.
// does not depend on this.
//
// The upshot is that this function computes (after checking for over-range/under-range):
//	uVolts = ((ADCResult [a 24-bit signed number]) * (-2500000)) / (2 ^ 24)
//
// The computation is carried out using 64-bit integer arithmetic to avoid loss of precision.
//
// ADCResult		the raw ADC reading
// ADCResultBits	the number of usable bits in ADCResult.
//				This should be the number of bits actually clocked out of the ADC, minus
//				any sub LSBs.  For the LTC2910, 27 is the "obvious" number.  As a practical
//				matter, given the scale factor of 2.5 million, numbers above 24 give the same answer.
// pOverUnder		place to return over-range/under-range indication
//				If the result was over-range or under-range, the function returns 0.
//
// *pOverUnder is 0 if the input was in range, -1 if under range, and 1 if over range.
// E.g.:
//	int OverUnder;
//	int uVolts = VNAccessADCResultTouVolts (ADCResult, ADCResultBits, &OverUnder);
//
VNACCESS_API int _stdcall VNAccessADCResultTouVolts (unsigned int ADCResult, unsigned int ADCResultBits, int *pOverUnder);

//
// This decomposes an ADCResult into raw count data and indicates over-range/under-range.
//
// This raw count data is a 32-bit signed integer that ranges from -(2^23) to (2^23)+1
//
// ADCResult		the raw ADC reading
// ADCResultBits	the number of usable bits in ADCResult.
//				This should be the number of bits actually clocked out of the ADC, minus
//				any sub LSBs.  For the LTC2910, 27 is the "obvious" number.
// pOverUnder		place to return over-range/under-range indication
//				If the result was over-range or under-range, the function returns 0.
//
// *pOverUnder is 0 if the input was in range, -1 if under range, and 1 if over range.
// E.g.:
//	int OverUnder;
//	int ADCCounts = VNAccessADCResultToCounts (ADCResult, ADCResultBits, &OverUnder);
//
VNACCESS_API int _stdcall VNAccessADCResultToCounts (unsigned int ADCResult, unsigned int ADCResultBits, int *pOverUnder);


//
// Computes the rectangular representation of a complex number (vector).
// This vector contains the components of the signal applied to the RF input of the VNA detector.
//
// See the "Detector" section (page 8) of Paul Kiciak's "An HF Vector Network Analyzer - Part 1".
//
// Note that this vector also contains the gain and phase components contributed by the detector itself.
//
// However, for reflection measurements, when the results of this function are interpreted as a "measured reflection
// coefficient" and used in the one-port calibration model, the detector components vanish in the computed actual result.
//
VNACCESS_API void _stdcall VNAccessRFVectorFromDetectorMeasurements (int dwVDC0, int dwVOFF0, int dwVDC90, int dwVOFF90, double *pRe, double *pIm);

#if 0
//
// Here is some sample code.
//
int VNAccessSample ()
{
	int r;
	unsigned int EC, dwADCResult;

	// Open the VNA.
	VNAHANDLE hVNA = VNAccessOpen ("LPT1", 0x3F8, 0, &EC);
	if (hVNA == VNACCESS_INVALIDHANDLE)
		return 1;

	// See if the VNA is "out there".
	//
	// We have determined the cleverest way is to start a conversion, get the result, and check for zero.
	//
	{
		// Get a conversion going - we are willing to busy wait 200ms.
		r = VNAccessStartADC (hVNA, 0, 0, 200000, NULL, VNACCESS_ADC_1);
		if (r == VNACCESS_SUCCESS || r == VNACCESS_DDSPOWEREDDOWN)
			; // continue
		else if (r == VNACCESS_VNANOTRESPONDING) {
			// report VNA not responding
			// This indicates the SDO line of the ADC is always high.
			// This often means that the VNA is plugged in but power is not applied.
			return 2;
		}
		else {
			// anything else is a hard error!
			return 1;
		}

		// Get an ADC Result - make sure it is not all zeros.
		r = VNAccessGetADCResult (hVNA, 200000, &Result, 32, NULL, 0, 0, NULL, VNACCESS_ADC_1, NULL);
		if (r == VNACCESS_SUCCESS || r == VNACCESS_DDSPOWEREDDOWN) {
			if (Result == 0) {
				// ADC SDO line appears to be grounded.
				// This usually means either the parallel port is not connected to the VNA or there is
				// a short.
				return 2;
			}
			continue;
		}
		else if (r == VNACCESS_VNANOTRESPONDING) {
			// report VNA not responding
			// This indicates the SDO line of the ADC is always high.
			// This often means that the VNA is plugged in but power is not applied.
			// Should have been caught above!
			return 2;
		}
		else {
			// anything else is a hard error!
			return 1;
		}
	}

	// Set the CFP mode.
	VNAccessSetDDSCFP (hVNA, TRUE);

	// Take a single measurement
	{
		//
		// Set the DDS state.
		// This can't fail for VNACCESS_CONVERSIONINPROGRESS.
		//
		VNAccessResetDDS (hVNA);	// Synchronizes DDS chips!
		unsigned int usTC;
		r = VNAccessSetDDS (hVNA, 0x80000000, 0x0, FALSE, 0x80000000, 0x0F, FALSE, &usTC);
		if (r != VNACCESS_SUCCESS)
			return 1;

		//
		// Start a conversion.
		// We busy wait for any current conversion to finish.
		// We specify 10ms of setup time for the applied DDS signal to work its way through the DUT
		// and back to the detector.
		//
		if (VNAccessStartADC (hVNA, usTC, 10000, 200000, NULL, VNACCESS_ADC_1) != VNACCESS_SUCCESS)
			return 1;

		// Sleep for some of the conversion time.
		::Sleep (120);

		//
		// Read the conversion result.
		// We busy wait for the conversion to finish.
		//
		if (VNAccessGetADCResult (hVNA, 200000, &dwADCResult, 32, NULL, 0, 0, NULL, VNACCESS_ADC_1, NULL) != VNACCESS_SUCCESS)
			return 1;
	}

	// Take a sequence of measurements (1.0 MHz - 10.0 MHz)
	{
		unsigned int f[10], v0[10], v90[10];
		unsigned int tw;
		unsigned int usDDSSetupMin = 10000;
		unsigned int usTC;
		int i;

		for (i=0 ; i < 10 ; i++) {
			v0[i] = v90[i] = 0;
			f[i] = (i+1) * 10000000 /*1.0 MHz*/;
		}
		tw = VNAccessFreqToTW (1483440000, f[0]);

		// Set up the DDS chips for measurement M0.
		// Perform a reset and power up, too.
		BOOL bSynced;
		r = VNAccessGetDDSSync (hVNA, &bSynced);
		if (r != VNACCESS_SUCCESS || !bSynced)
			VNAccessResetDDS (hVNA);	// Synchronizes the DDS chips!
		r = VNAccessSetDDS (hVNA, tw, 0x0, FALSE, tw, 0x0, FALSE, &usTC);
		if (r != VNACCESS_SUCCESS)
			return 1;

		// Start a 0-degree conversion - but not before our DDS Setup time has passed!
		r = VNAccessStartADC (hVNA, usTC, usDDSSetupMin, 200000, &usTC, VNACCESS_ADC_1);
		if (r != VNACCESS_SUCCESS)
			return 1;

		// Do the loop
		for (i=0 ; i < 10 ; i++) {
			int bLastTime = (i == 9);
			unsigned int v;
			int j;
#define N_AVG	4

			// Averaging loop - all but the last ADC reading.
			for (j=0 ; j < N_AVG-1 ; j++) {
				// Wait for conversion to finish.
				// Sleep (133 - (VNAccessGetusTickCount () - usTC)/1000);

				// Read the 0-degree result, start next conversion ASAP
				VNAccessGetADCResult (hVNA, 200000, &v, 32, NULL, 0, 0, &usTC, VNACCESS_ADC_1, NULL);
				v0[i] += v;
			}

			// While the (last) conversion happens, set the DDS shift registers
			// with new (90-degree) state (don't clock into core)
			VNAccessSetDDS (hVNA, tw, 0x0, FALSE, tw, 8, FALSE, VNACCESS_SETDDS_NO_FQ_UD);

			// Wait for conversion to finish.
			// Sleep (133 - (VNAccessGetusTickCount () - usTC)/1000);

			// Make sure the ADC result is ready (busy wait if necessary)...
			if (VNAccessStatus (hVNA, 200000, VNACCESS_ADC_1) != VNACCESS_SUCCESS)
				return 1;

			// Clock the DDS shift registers into the core...
			VNAccessDDSFQ_UD (hVNA, &usTC);

			// Wait for DDS-to-Detector setup time to pass
			// Sleep (usDDSSetupMin - (VNAccessGetusTickCOunt () - usTC)/1000);

			// Read the 0-degree ADC result.
			// (busy) wait, if necessary, for DDS-to-Detector time before starting 90-degree conversion
			VNAccessGetADCResult (hVNA, 0, &v, 32, NULL, usTC, usDDSSetupMin, &usTC, VNACCESS_ADC_1, NULL);

			// Do the averaging
			v0[i] += v;
			v0[i] /= N_AVG;


			// Averaging loop - all but the last ADC reading.
			for (j=0 ; j < N_AVG-1 ; j++) {
				// Wait for conversion to finish.
				// Sleep (133 - (VNAccessGetusTickCount () - usTC)/1000);

				// Read the 90-degree result, start next conversion ASAP
				VNAccessGetADCResult (hVNA, 200000, &v, 32, NULL, 0, 0, &usTC, VNACCESS_ADC_1, NULL);
				v90[i] += v;
			}

			// While the (last) conversion happens, set the DDS shift registers
			// with new (next freq., 0-degree) state (don't clock into core)
			// Compute new tuning word.
			if (!bLastTime) {
				tw = VNAccessFreqToTW (1483440000, f[i+1]);
				VNAccessSetDDS (hVNA, tw, 0x0, FALSE, tw, 0, FALSE, VNACCESS_SETDDS_NO_FQ_UD);
			}

			// Wait for conversion to finish.
			// Sleep (133 - (VNAccessGetusTickCount () - usTC)/1000);

			// Make sure the ADC result is ready (busy wait if necessary)...
			if (VNAccessStatus (hVNA, 200000, VNACCESS_ADC_1) != VNACCESS_SUCCESS)
				return 1;

			// Clock the DDS shift registers into the core...
			if (!bLastTime)
				VNAccessDDSFQ_UD (hVNA, &usTC);

			// Wait for DDS-to-Detector setup time to pass
			// Sleep (usDDSSetupMin - (VNAccessGetusTickCOunt () - usTC)/1000);

			// Read the 90-degree ADC result.
			// (busy) wait, if necessary, for DDS-to-Detector time before starting next freq., 0-degree conversion
			VNAccessGetADCResult (hVNA, 0, &v, 32, NULL, usTC, (bLastTime) ? 0 : usDDSSetupMin, &usTC, VNACCESS_ADC_1, NULL);

			// Do the averaging
			v90[i] += v;
			v90[i] /= N_AVG;
		}
	}

	// Close the VNA
	r = VNAccessClose (hVNA);
	if (r != VNACCESS_SUCCESS)
		return 1;

	return 0;
}
#endif
