// spectrum.cpp : implementation file
//
//(C) Copyright Dave Roberts G8KBB 2005
// This program is released only for the purpose of self training and eduation
// in amatuer radio. It may not be used for any commercial purpose, sold or
// modified.
// It has been written as an exercise in self tuition by me as part of my hobby
// and I make no claims about its fitness for purpose or correct operation.

#include "stdafx.h"
#include "NoiseMeter.h"
#include "measure.h"

#include "high-pass-filters.h"

#include "mmsystem.h"
#include "spectrum.h"
#include "setup.h"

#include "common.h"

#include "math.h"

#include "rfftw.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

extern bool bWrappedBuffer;
extern spectrum *pSpectrum;
extern setup *pSetup;
extern measure *pMeasure;

void CALLBACK waveGotBlockSpectrum(
  HWAVEIN hwi,       
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2     
);

void create_window( int uWindowSize);

fftw_real dSample[MAX_FFT_SAMPLE];
fftw_real out[MAX_FFT_SAMPLE], power_spectrum[MAX_AVERAGE+1][MAX_FFT_SAMPLE/2+1];
rfftw_plan p;
bool rfftw_plan_valid;
BOOL bGotPowerSpectrum;
double dWindow[MAX_FFT_SAMPLE];

int nPsIndex;

// used to define spectrum trace display area
CRect RectScan;

/////////////////////////////////////////////////////////////////////////////
// spectrum property page

IMPLEMENT_DYNCREATE(spectrum, CPropertyPage)

spectrum::spectrum() : CPropertyPage(spectrum::IDD)
{
	//{{AFX_DATA_INIT(spectrum)
	m_nDbPerDivCombo = -1;
	m_nDbTopCombo = -1;
	//}}AFX_DATA_INIT
	m_nVerticalDivisions = -1;
	m_ndBTop = -1;
	rfftw_plan_valid = FALSE;
}

spectrum::~spectrum()
{
	StopInput();

	if( rfftw_plan_valid )
		rfftw_destroy_plan(p);
	rfftw_plan_valid = FALSE;
}

void spectrum::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(spectrum)
	DDX_CBIndex(pDX, IDC_COMBO1, m_nDbPerDivCombo);
	DDX_CBIndex(pDX, IDC_COMBO2, m_nDbTopCombo);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(spectrum, CPropertyPage)
	//{{AFX_MSG_MAP(spectrum)
	ON_WM_PAINT()
	ON_CBN_SELCHANGE(IDC_COMBO1, OnSelchangeVerticalDivisions)
	ON_CBN_SELCHANGE(IDC_COMBO2, OnSelchangedBTop)
	ON_CBN_SELCHANGE(IDC_COMBO_HORIZ_SCAN, OnSelchangeComboHorizScan)
	ON_BN_CLICKED(IDC_CHECK_S_AVERAGE, OnCheckSAverage)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// spectrum message handlers

// ********************************************************************************
// There are 2 main workhorses. This is one, the painting of the screen. The other
// is the WaveInProc() handler.
// We draw a grid, label it and then do a spectrum display if one is available.

void spectrum::OnPaint() 
{
	char szTemp[16];
	int j;
	int nBlockLength;
	double dMaxFreq;

//	CWnd* pWnd = GetDlgItem(IDC_EDIT_DISP);
	
	CPaintDC dc(this);
	// wide pen for border
	CPen penSolidBlackWide( PS_SOLID, 2, RGB(0,0,0) );
	CPen penSolidGreyNarrow( PS_SOLID, 2, RGB(192,192,192) );
	CPen penSolidRedNarrow( PS_SOLID, 1, RGB(255,0,0) );
	// trace is our display area
	CRect trace = RectScan;
	// frig size to multiple of divisions
	trace.right = trace.left + ((int)(trace.right - trace.left)/DEFAULT_HORIZONTAL_DIVISIONS)*DEFAULT_HORIZONTAL_DIVISIONS;
	trace.bottom = trace.top + ((int)(trace.bottom - trace.top)/m_nVerticalDivisions)*m_nVerticalDivisions;
	dc.FillRect( trace, CBrush::FromHandle((HBRUSH)GetStockObject( WHITE_BRUSH)) );
	// height, width, top and left coords
	int h = trace.Height();
	int w = trace.Width();
	int l = trace.TopLeft().x;
	int t = trace.TopLeft().y;
	// select wide pen and draw border
	CPen *pOldPen = dc.SelectObject( &penSolidBlackWide );
	dc.MoveTo( l,t);
	dc.LineTo( l,t+h);
	dc.LineTo( l+w,t+h);
	dc.LineTo( l+w,t);
	dc.LineTo( l,t);
	// narrow pen and delete old wide pen
	dc.SelectObject( &penSolidGreyNarrow );
	penSolidBlackWide.DeleteObject();
	// draw grid
	for( int i=h/m_nVerticalDivisions; i<h; i+=h/m_nVerticalDivisions)
	{
		dc.MoveTo( l+1, t+i );
		dc.LineTo( l+w-2, t+i);
	}
	for( i=w/DEFAULT_HORIZONTAL_DIVISIONS; i<w; i+=w/DEFAULT_HORIZONTAL_DIVISIONS)
	{
		dc.MoveTo( l+i, t+1 );
		dc.LineTo( l+i, t+h-2);
	}
	// LABEL VERTICAL SCALE
	dc.SetTextAlign(TA_RIGHT);
	for( i=0, j=0; i<h; j-=10, i+=h/m_nVerticalDivisions)
		dc.TextOut( l-5, t+i-8, _itoa( m_ndBTop+j, szTemp, 10 ) );

	dc.SelectObject( &penSolidRedNarrow );
	penSolidGreyNarrow.DeleteObject();

	switch( m_nHorizontalSelection )
	{
		case 0:
			nBlockLength = pSetup->nFFTBlockSize/2;
			dMaxFreq = pSetup->nSampleRate/2;
			break;
		case 1:
			dMaxFreq = 4000;
			nBlockLength = (int)(dMaxFreq/pSetup->nSampleRate * pSetup->nFFTBlockSize ); // /2);
			break;
		case 2:
			dMaxFreq = 1000;
			nBlockLength = (int)(dMaxFreq/pSetup->nSampleRate * pSetup->nFFTBlockSize ); // /2);
			break;
		default:
			ASSERT(FALSE);
	}
	sprintf( szTemp, "%.1f", dMaxFreq/1000 );
	dc.TextOut( l+w, t+h+8, szTemp );
	dc.TextOut( l+w/2, t+h+8, "KHz" );
	dc.TextOut( l, t+h+8, "0" );
	dc.TextOut( l-30, t+h/2+6, "dB" );


	if( bGotPowerSpectrum )
	{
		j = (int)(t+(m_ndBTop-power_spectrum[0][0])*h/10/m_nVerticalDivisions);
		if( j < t ) j = t;
		if( j > t+h) j = t+h;
		dc.MoveTo(l,j);
		for(i=1;i<nBlockLength;i++)
		{
			j = (int)(t+(m_ndBTop-power_spectrum[0][i])*h/10/m_nVerticalDivisions);
			if( j < t ) j = t;
			if( j > t+h) j = t+h;
			dc.LineTo(l+(i*w)/nBlockLength, j );
		}
	}

	dc.SelectObject( pOldPen );
	penSolidRedNarrow.DeleteObject();

	ReleaseDC( &dc );
	// Do not call CPropertyPage::OnPaint() for painting messages
}

// ********************************************************************************
// Initialisation.
// Populate drop downs etc.

BOOL spectrum::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	char szTemp[16];

	CWnd* pWnd = GetDlgItem(IDC_COMBO1);
	ASSERT_VALID(pWnd);

	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	for( int nCount =4; nCount <= 12; nCount++)
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)_itoa(nCount,szTemp,10)));
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)DEFAULT_VERTICAL_DIVISIONS-4, 0);
	m_nVerticalDivisions = DEFAULT_VERTICAL_DIVISIONS;

	pWnd = GetDlgItem(IDC_COMBO2);
	ASSERT_VALID(pWnd);

	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	for( nCount =0; nCount >= -80; nCount-=10)
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)_itoa(nCount,szTemp,10)));
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)0, 0);
	m_ndBTop = 0;

	m_nHorizontalSelection = 0;

	pWnd = GetDlgItem(IDC_COMBO_HORIZ_SCAN);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)"Full Scan"));
	pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)"0 - 4 KHz"));
	pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)"0 - 1 KHz"));
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)0, 0);

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}

// ********************************************************************************
// Handle user change to number of vertical divisions to the display

void spectrum::OnSelchangeVerticalDivisions() 
{
	CWnd* pWnd = GetDlgItem(IDC_COMBO1);
	ASSERT_VALID(pWnd);

	m_nVerticalDivisions = pWnd->SendMessage(CB_GETCURSEL, 0, 0) + 4;

	RedrawWindow();
}

// ********************************************************************************
// Handle change to the dB level at the top of the spectrum display

void spectrum::OnSelchangedBTop() 
{
	CWnd* pWnd = GetDlgItem(IDC_COMBO2);
	ASSERT_VALID(pWnd);

	m_ndBTop = pWnd->SendMessage(CB_GETCURSEL, 0, 0)*(-10);

	RedrawWindow();
	
}

// ********************************************************************************
// When we become active:
//		Create a "plan" for FFT (see rfftw documentation)
//		Create a hanning window for the FFT data
//		Set up & start the audio data
//			and show the an error if we can't start the audio process

BOOL spectrum::OnSetActive() 
{
	bGotPowerSpectrum = FALSE;
	CException ex;

	p = rfftw_create_plan(pSetup->nFFTBlockSize, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
	rfftw_plan_valid = TRUE;

	create_window(pSetup->nFFTBlockSize);

	nPsIndex = 1;
	bWrappedBuffer = FALSE;
	// set average flag
	OnCheckSAverage();

	set_wfx();

	ASSERT( !bRunning);
	if( IsAudioDeviceValid() )
		if( StartReading((DWORD)waveGotBlockSpectrum, pSetup->nFFTBlockSize * wfx.nBlockAlign)!= MMSYSERR_NOERROR )
				ex.ReportError( MB_OK, IDS_CANNOT_OPEN_WAVEIN );

	// find the display area as we will normally only invalidate
	// this in each scan
	pSpectrum->GetClientRect( &RectScan );
	RectScan.left += LEFT_OFFSET;
	RectScan.right -= RIGHT_OFFSET;
	RectScan.top += TOP_OFFSET;
	RectScan.bottom -= BOTTOM_OFFSET;

	return CPropertyPage::OnSetActive();
}

// ********************************************************************************
// This is the WaveInProc() for spectrum analysis.
// Capture the audio, do an FFT, create the power spectrum and send ourselves a
// message to repaint the window.

void CALLBACK waveGotBlockSpectrum(
  HWAVEIN hwi,       
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2 )    
{
	__int32 x;

	if( uMsg == WIM_DATA && bRunning == TRUE)
	{
		SHORT *ptr = (short *)(((LPWAVEHDR)dwParam1)->lpData);
		unsigned char *ptrc = (unsigned char *)(((LPWAVEHDR)dwParam1)->lpData);
		DWORD i,j;
		int nAdd2ForStereo = wfx.nChannels == 1 ? 1 : 2;

		FilterInit( dSample[0] );

		DWORD dwBytesRecorded = ((LPWAVEHDR)dwParam1)->dwBytesRecorded;
		fftw_real *dPtr=&dSample[0];
		int k = 0;
		switch( wfx.wBitsPerSample )
		{
			case 24:
				j = 3 * nAdd2ForStereo;
				for( i = 0; i < dwBytesRecorded && k < pSetup->nFFTBlockSize; k++, i+= j, ptrc += 3*nAdd2ForStereo)
				{
					x = ptrc[0] + (ptrc[1]<<8) + (ptrc[2]<<16) +( ptrc[2] &0x80? 255<<24: 0 );
					*dPtr++ = ((double)(x))/(32768*256) * dWindow[k];
				}
				break;
			case 16:
				j = 2 * nAdd2ForStereo;
				for( i = 0; i < dwBytesRecorded && k < pSetup->nFFTBlockSize; k++, i+= j, ptr += nAdd2ForStereo)
				{
					*dPtr++ = (double)(*ptr)/32768 * dWindow[k];
				}
				break;
			case 8:
				j = nAdd2ForStereo;
				for( i = 0; i < dwBytesRecorded && k < pSetup->nFFTBlockSize; k++, i += j, ptrc += nAdd2ForStereo)
				{
					*dPtr++ = (double)(*ptrc-128)/128  * dWindow[k];
				}
				break;
			default:
				ASSERT( FALSE );
		}

		if( pSetup->bUseHighPassFilter )
			for(k=0;k<pSetup->nFFTBlockSize; k++)
				dSample[k] = (*pFilter)( dSample[k] );

		rfftw_one(p, dSample, out);
		power_spectrum[nPsIndex][0] = out[0]*out[0];  /* DC component */
		for (k = 1; k < (pSetup->nFFTBlockSize+1)/2; ++k)  /* (k < N/2 rounded up) */
			power_spectrum[nPsIndex][k] = out[k]*out[k] + out[pSetup->nFFTBlockSize-k]*out[pSetup->nFFTBlockSize-k];
		if (pSetup->nFFTBlockSize % 2 == 0) /* N is even */
			power_spectrum[nPsIndex][pSetup->nFFTBlockSize/2] = out[pSetup->nFFTBlockSize/2]*out[pSetup->nFFTBlockSize/2];  /* Nyquist freq. */
		for(k=0;k<(pSetup->nFFTBlockSize+1)/2;k++)
			power_spectrum[nPsIndex][k] = 10*log10(power_spectrum[nPsIndex][k]/pSetup->nFFTBlockSize);
		if (pSetup->nFFTBlockSize % 2 == 0) /* N is even */
			power_spectrum[nPsIndex][pSetup->nFFTBlockSize/2] = 10*log10(power_spectrum[nPsIndex][pSetup->nFFTBlockSize/2]/pSetup->nFFTBlockSize);
		MMRESULT mmResult = waveInAddBuffer( hwi, (LPWAVEHDR)dwParam1, sizeof( WAVEHDR) );
		bGotPowerSpectrum = TRUE;

		// we always display trace 0.
		// so if we are not averaging, just copy the current
		// trace into trace 0.
		// If we are averaging, then average the traces and
		// put that in trace 0.
		if( !pSpectrum->bAverageSpectrum )
		{
			for( k=0; k< (pSetup->nFFTBlockSize+1)/2; k++)
				power_spectrum[0][k] = power_spectrum[nPsIndex][k];
		}
		else
		if( nPsIndex > 0 )
		{
			int i,j; 

			// How mch data? If we have wrapped the buffer we have a full
			// set of traces, otherwise use the number we have.
			k  = bWrappedBuffer ? pMeasure->m_nAverageCount : nPsIndex;
			for( i=0; i< (pSetup->nFFTBlockSize+1)/2; i++)
			{	for( power_spectrum[0][i] = 0, j=1; j <= k; j++)
					power_spectrum[0][i] += power_spectrum[j][i];
				power_spectrum[0][i] /= k;
			}
		}
		// bump the counter and wrap if we have reached desrie count
		nPsIndex++;
		if( nPsIndex > pMeasure->m_nAverageCount )
		{
			nPsIndex = 1;
			bWrappedBuffer = TRUE;
		}

		pSpectrum->InvalidateRect(&RectScan, TRUE);
		k = pSpectrum->SendMessage(WM_PAINT,0,0);
	}
}

// ********************************************************************************
// When page no longer active, get rid of the plan for the FFT code as we may need
// a different one next time we run. Also turn off audio processes.

BOOL spectrum::OnKillActive() 
{
	StopInput();

	rfftw_destroy_plan(p);
	rfftw_plan_valid = FALSE;

	return CPropertyPage::OnKillActive();
}

// ********************************************************************************
// Build a window to apply to the data before FFT.
// The only one we support here is Hanning. It is generated on the fly depending
// on the window size. We do this when the page becomes active.

void create_window( int uWindowSize)
{
	if( uWindowSize <= MAX_FFT_SAMPLE )
	{
		for(int i=0; i < uWindowSize; i++)
			dWindow[i] = (double)0.5 - 0.5*(cos(2*PI*i/(uWindowSize-1)));
	}
};

// ********************************************************************************
// User wants to select a new horizontal scan. Read selection and store it.

void spectrum::OnSelchangeComboHorizScan() 
{
	CWnd *pWnd = GetDlgItem(IDC_COMBO_HORIZ_SCAN);
	ASSERT_VALID(pWnd);
	m_nHorizontalSelection = pWnd->SendMessage(CB_GETCURSEL, 0, 0);	
	RedrawWindow();
}

// ********************************************************************************
// User wants to change state - average or do not average traces.

void spectrum::OnCheckSAverage() 
{
	CWnd *pWnd = GetDlgItem(IDC_CHECK_S_AVERAGE);
	ASSERT_VALID(pWnd);
	bAverageSpectrum = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;	
}
