// setup.cpp : implementation file
//
//(C) Copyright Dave Roberts G8KBB 2005
// This program is released only for the purpose of self training and eduation
// in amatuer radio. It may not be used for any commercial purpose, sold or
// modified.
// It has been written as an exercise in self tuition by me as part of my hobby
// and I make no claims about its fitness for purpose or correct operation.

#include "stdafx.h"
#include "NoiseMeter.h"
#include "setup.h"

#include "mmsystem.h"

#include "math.h"
#include "string.h"

#include "spectrum.h"

#include "high-pass-filters.h"

#include "common.h"

#include "objbase.h"
#include "iostream.h"
#include "vnaio.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// This is used to eliminate 8 bit ADC settings from the available options
#define EXCLUDE_8_BIT

// When an ENRcalibration file is read in, it is parsed and stored in this table
// to allow it to be used in determining an ENR from a frequency.
// To speed this up, the file is sorted into ascending frequencies.
// The actual table size is stored in the variable nCalibrationTableSize
struct cal_table
{
	double dFreq;
	double dEnr;
} CalibrationTable[CAL_TABLE_SIZE];

int nCalibrationTableSize;

// The following table is used to find device capabilities. For each bit value
// as defined in mmsystem.h (such as WAVE_FORMAT_1M16) we define the relevant
// parameters for the program to use.

struct format_table
{
	DWORD	formatid;
	int		nName;
	int		nBitsPerSample;
	int		nSampleRate;
	bool	bIsStereo;
} Formats[] = 
{
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_1M08, IDS_STRING_1M08, 8,  11025, FALSE },
#endif
	{ WAVE_FORMAT_1M16, IDS_STRING_1M16, 16, 11025, FALSE },
	{ -1, IDS_STRING_1M24, 24, 11025, FALSE },
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_1S08, IDS_STRING_1S08, 8,  11025, TRUE  },
#endif
	{ WAVE_FORMAT_1S16, IDS_STRING_1S16, 16, 11025, TRUE  },
	{ -1, IDS_STRING_1S24, 24, 11025, TRUE  },
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_2M08, IDS_STRING_2M08, 8,  22050, FALSE },
#endif
	{ WAVE_FORMAT_2M16, IDS_STRING_2M16, 16, 22050, FALSE },
	{ -1, IDS_STRING_2M24, 24, 22050, FALSE },
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_2S08, IDS_STRING_2S08, 8,  22050, TRUE  },
#endif
	{ WAVE_FORMAT_2S16, IDS_STRING_2S16, 16, 22050, TRUE  },
	{ -1, IDS_STRING_2S24, 24, 22050, TRUE  },
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_4M08, IDS_STRING_4M08, 8,  44100, FALSE },
#endif
	{ WAVE_FORMAT_4M16, IDS_STRING_4M16, 16, 44100, FALSE },
	{ -1, IDS_STRING_4M24, 24, 44100, FALSE },
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_4S08, IDS_STRING_4S08, 8,  44100, TRUE  },
#endif
	{ WAVE_FORMAT_4S16, IDS_STRING_4S16, 16, 44100, TRUE  },
	{ -1, IDS_STRING_4S24, 24, 44100, TRUE  },
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_48M08, IDS_STRING_48M08, 8,  48000, FALSE },
#endif
	{ WAVE_FORMAT_48M16, IDS_STRING_48M16, 16, 48000, FALSE },
	{ -1, IDS_STRING_48M24, 24, 48000, FALSE },
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_48S08, IDS_STRING_48S08, 8,  48000, TRUE  },
#endif
	{ WAVE_FORMAT_48S16, IDS_STRING_48S16, 16, 48000, TRUE  },
	{ -1, IDS_STRING_48S24, 24, 48000, TRUE  },
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_96M08, IDS_STRING_96M08, 8,  96000, FALSE },
#endif
	{ WAVE_FORMAT_96M16, IDS_STRING_96M16, 16, 96000, FALSE },
	{ -1, IDS_STRING_96M24, 24, 96000, FALSE },
#ifndef EXCLUDE_8_BIT
	{ WAVE_FORMAT_96S08, IDS_STRING_96S08, 8,  96000, TRUE  },
#endif
	{ WAVE_FORMAT_96S16, IDS_STRING_96S16, 16, 96000, TRUE  },
	{ -1, IDS_STRING_96S24, 24, 96000, TRUE  },
	{ 0,0,0,0, FALSE }
};

// function definition for private function to help sort the calibration
// table. Called from qsort.

int CalibrationTableSortCompare( const void *arg1, const void *arg2 );

/////////////////////////////////////////////////////////////////////////////
// setup property page

IMPLEMENT_DYNCREATE(setup, CPropertyPage)

setup::setup() : CPropertyPage(setup::IDD)
{
	//{{AFX_DATA_INIT(setup)
	//}}AFX_DATA_INIT
	
	WAVEINCAPS wic;
	CString sTempDeviceName;
	CString sSetup, sToken;
	sSetup.LoadString(IDS_STRING_SETUP);
	hAutomateDevice = INVALID_HANDLE_VALUE;
	bUseLogFile = FALSE;

	sToken.LoadString(IDS_WAVE_IN_DEVICE);
	m_uWaveInDev = AfxGetApp( )->GetProfileInt( sSetup, sToken, 0 );
	sToken.LoadString(IDS_WAVE_IN_FORMAT);
#ifdef EXCLUDE_8_BIT
	m_uWaveInFormat = AfxGetApp( )->GetProfileInt( sSetup, sToken, 8 );
#else
	m_uWaveInFormat = AfxGetApp( )->GetProfileInt( sSetup, sToken, 12 );
#endif
	if( m_uWaveInFormat >= (sizeof( Formats ) / sizeof(format_table)) -1)
		m_uWaveInFormat = -1;
	nSampleRate = Formats[m_uWaveInFormat].nSampleRate;
	nBitsPerSample = Formats[m_uWaveInFormat].nBitsPerSample;
	bIsStereo = Formats[m_uWaveInFormat].bIsStereo;
	nMilliSeconds = 1000;
	SetFilterRate( nSampleRate );

	sToken.LoadString(IDS_FFT_BLOCK_SIZE);
	nFFTBlockSize = AfxGetApp( )->GetProfileInt( sSetup, sToken, DEFAULT_FFT_BLOCK_SIZE );
	if( nFFTBlockSize > MAX_FFT_SAMPLE )
		nFFTBlockSize = MAX_FFT_SAMPLE;

	sToken.LoadString(IDS_ENR_CAL_FILE);
	m_Filename = AfxGetApp( )->GetProfileString( sSetup, sToken, "" );

	sToken.LoadString(IDS_LOG_FILE);
	m_LogFilename = AfxGetApp( )->GetProfileString( sSetup, sToken, "" );

	sToken.LoadString(IDS_AUTOMATE_NAME);
	szAutomateName = AfxGetApp( )->GetProfileString( sSetup, sToken, "" );
	if( strlen( szAutomateName ) > 0 )
		bEnableAutomation = strcmp( szAutomateName, "None") == 0 ?  FALSE : TRUE;

	sToken.LoadString(IDS_STRING_USE_HIGH_PASS_FILTER);
	bUseHighPassFilter = AfxGetApp( )->GetProfileInt( sSetup, sToken, 0 ) != 0 ? TRUE : FALSE;

	sToken.LoadString(IDS_AUTOMATE_NOISE_SELECTION);
	nAutomateNoiseSelection = AfxGetApp( )->GetProfileInt( sSetup, sToken, 0 );

	sToken.LoadString(IDS_AUTOMATE_DUT_SELECTION);
	nAutomateDutSelection = AfxGetApp( )->GetProfileInt( sSetup, sToken, 0 );

	sToken.LoadString(IDS_AUTOMATE_DECIMALS);
	nAutomateDecimalPoints = AfxGetApp( )->GetProfileInt( sSetup, sToken, 3 );

	sToken.LoadString(IDS_USB_DEVICE);
	m_UsbDeviceName = AfxGetApp( )->GetProfileString( sSetup, sToken, "ezusb-0" );
	if( strlen(m_UsbDeviceName) >= 256)
		m_UsbDeviceName = "ezusb-0";

	sToken.LoadString(IDS_WAVE_IN_DEVICE_NAME);
	if( waveInGetDevCaps( m_uWaveInDev, &wic, (UINT)sizeof(WAVEINCAPS)) == MMSYSERR_NOERROR )
	sTempDeviceName = AfxGetApp( )->GetProfileString( sSetup, sToken, "None" );
	if( strcmp( wic.szPname, sTempDeviceName ) != 0 )
		m_uWaveInDev = -1;

	sToken.LoadString(IDS_USE_ENR_FILE);
	bUseEnrFile = AfxGetApp( )->GetProfileInt( sSetup, sToken, 0 ) != 0 ? TRUE : FALSE;
	if( bUseEnrFile )
	{
		if( ReadValidEnrFile() != 0 )
		{
			nCalibrationTableSize = 0;
			bUseEnrFile = FALSE;
		}
	}
}

setup::~setup()
{
	if( hAutomateDevice != INVALID_HANDLE_VALUE )
	{
		CloseHandle( hAutomateDevice );
		hAutomateDevice = INVALID_HANDLE_VALUE;
	}

}

void setup::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(setup)
	DDX_CBIndex(pDX, IDC_COMBO1, m_uWaveInDev);
	DDX_CBIndex(pDX, IDC_COMBO2, m_uWaveInFormat);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(setup, CPropertyPage)
	//{{AFX_MSG_MAP(setup)
	ON_CBN_SELENDOK(IDC_COMBO1, OnSelendokWaveInDev)
	ON_CBN_SELENDOK(IDC_COMBO2, OnSelendokWaveInFormat)
	ON_BN_CLICKED(IDC_BUTTON1, OnButtonSetupSave)
	ON_CBN_SELENDOK(IDC_COMBO_TIME, OnSelendokComboTime)
	ON_BN_CLICKED(IDC_BUTTON_FILE, OnButtonFile)
	ON_EN_KILLFOCUS(IDC_EDIT_CONFIG_FILENAME, OnKillfocusEditConfigFilename)
	ON_CBN_SELENDOK(IDC_COMBO_FFT_BLOCK, OnSelendokComboFftBlock)
	ON_CBN_SELENDOK(IDC_COMBO_AUTO, OnSelendokComboAuto)
	ON_BN_CLICKED(IDC_CHECK_USE_ENR_FILE, OnCheckUseEnrFile)
	ON_CBN_SELENDOK(IDC_COMBO_NOISE, OnSelendokComboNoise)
	ON_BN_CLICKED(IDC_BUTTON_LOG_FILE, OnButtonLogFile)
	ON_EN_KILLFOCUS(IDC_EDIT_LOG_FILENAME, OnKillfocusEditLogFilename)
	ON_BN_CLICKED(IDC_CHECK_LOG_RESULTS, OnCheckLogResults)
	ON_BN_CLICKED(IDC_CHECK_HIGH_PASS, OnCheckHighPass)
	ON_CBN_SELENDOK(IDC_COMBO_DUT, OnSelendokComboDut)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// setup message handlers

BOOL setup::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	// TODO: Add extra initialization here

	WAVEINCAPS wic;
	char szTemp[128];
	HANDLE hTemp;

	// GET SOUND INPUT DEVICE DETAILS AND POPULATE COMBO
	UINT uNumDevices = waveInGetNumDevs();
	
	CWnd* pWnd = GetDlgItem(IDC_COMBO1);
	ASSERT_VALID(pWnd);

	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	for( UINT uCount =0; uCount < uNumDevices; uCount++)
		if( waveInGetDevCaps( uCount, &wic, (UINT)sizeof(WAVEINCAPS)) == MMSYSERR_NOERROR )
			pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)wic.szPname));
		else
			pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)"Unknown Device");
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)m_uWaveInDev, 0);
	PopulateCombo2();

	pWnd = GetDlgItem(IDC_COMBO_FFT_BLOCK);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	for( uCount =10; uCount <= MAX_FFT_SAMPLE_EXPONENT; uCount++)
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)_itoa((int)(pow(2,uCount)),szTemp,10)));
	_itoa( nFFTBlockSize, szTemp, 10);
	int i = pWnd->SendMessage(CB_FINDSTRINGEXACT, (WPARAM)-1, (LPARAM)szTemp);
	if( i == CB_ERR )
		i = 0;
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)i, 0);

	ASSERT( MAX_FFT_SAMPLE == pow(2,uCount-1));

	pWnd = GetDlgItem(IDC_COMBO_TIME);
	ASSERT_VALID(pWnd);

	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)"10 seconds"));
	pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)"1 second"));
	pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)"100 msec"));
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)1, 0);

	pWnd = GetDlgItem(IDC_EDIT_CONFIG_FILENAME);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText( m_Filename );

	pWnd = GetDlgItem(IDC_EDIT_LOG_FILENAME);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText( m_LogFilename );

	pWnd = GetDlgItem(IDC_COMBO_AUTO);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)"None");
	int nDesiredSelection = 0;
	int nSelectionCount = 1; // "none" above
	for(i=1; i<8; i++)
	{
		sprintf(szTemp,"COM%d",i);
		if (      (hTemp = CreateFile( szTemp, 0,
                  0,                    // exclusive access
                  NULL,                 // no security attrs
                  OPEN_EXISTING,
                  FILE_ATTRIBUTE_NORMAL, NULL )) != (HANDLE) -1 )
		{
			pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)szTemp);
			CloseHandle( hTemp );
			if( strcmp( szTemp, szAutomateName ) == 0 )
				nDesiredSelection = nSelectionCount;
			nSelectionCount++;
		}

	}
	for(i=1; i<8; i++)
	{
		sprintf(szTemp,"LPT%d",i);
		if (      (hTemp = CreateFile( szTemp, 0,
                  0,                    // exclusive access
                  NULL,                 // no security attrs
                  OPEN_EXISTING,
                  FILE_ATTRIBUTE_NORMAL, NULL )) != (HANDLE) -1 )
		{
			pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)szTemp);
			CloseHandle( hTemp );
			if( strcmp( szTemp, szAutomateName ) == 0 )
				nDesiredSelection = nSelectionCount;
			nSelectionCount++;
		}
	}
	pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)"USB");
	if( strcmp( "USB", szAutomateName ) == 0 )
		nDesiredSelection = nSelectionCount;

	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)nDesiredSelection, 0);

	pWnd = GetDlgItem(IDC_CHECK_USE_ENR_FILE);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(BM_SETCHECK, bUseEnrFile ? BST_CHECKED :BST_UNCHECKED, 0);
	
	pWnd = GetDlgItem(IDC_CHECK_HIGH_PASS);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(BM_SETCHECK, bUseHighPassFilter ? BST_CHECKED :BST_UNCHECKED, 0);

	PopulateComboNoiseDUT();

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}

// ********************************************************************************
// This is a helper utility called within setup.
// Given a setting for the automatic switching device, populate the two function
// specific drop downs with the appropriate options for that device. 
// For COM devices, this is RTS and DTR. For LPT it will be D0..D7 but LPT is not
// supported yet. For N2PK VNA USB interface it will be Port D bits 0..7

void setup::PopulateComboNoiseDUT()
{
	char szTemp[128];

	CWnd *pWnd = GetDlgItem(IDC_COMBO_AUTO);
	ASSERT_VALID(pWnd);
	int i = pWnd->SendMessage(CB_GETCURSEL, 0, 0);
	pWnd->SendMessage(CB_GETLBTEXT, i, (WPARAM)szTemp);
	szAutomateName = szTemp;
	bEnableAutomation = strcmp( szAutomateName, "None") == 0 ?  FALSE : TRUE;

	pWnd = GetDlgItem(IDC_COMBO_NOISE);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	if( strncmp( szAutomateName, "COM", 3 ) == 0 )
	{
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)"RTS");
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)"DTR");
	}
	else if( strncmp( szAutomateName, "USB", 3 ) == 0 )
	{
		strcpy( szTemp, "D " );
		for(i=0; i<8;i++)
		{
			szTemp[1] = i+'0';
			pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)szTemp);
		}
	}

	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)nAutomateNoiseSelection, 0);

	pWnd = GetDlgItem(IDC_COMBO_DUT);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	if( strncmp( szAutomateName, "COM", 3 ) == 0 )
	{
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)"RTS");
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)"DTR");
		nAutomateDutSelection = nAutomateNoiseSelection == 0 ? 1 : 0;
		pWnd->EnableWindow( FALSE );
	}
	else if( strncmp( szAutomateName, "USB", 3 ) == 0 )
	{
		strcpy( szTemp, "D " );
		for(i=0; i<8;i++)
		{
			szTemp[1] = i+'0';
			pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)szTemp);
		}
		pWnd->EnableWindow( TRUE );
	}
	else
	{
		pWnd->EnableWindow( TRUE );
		nAutomateDutSelection = 0;
	}
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)nAutomateDutSelection, 0);
}

// ********************************************************************************
// This is a function called by other routines in setup just to reuse common code
// It populates the wave formats drop down with new entries depending on the contents
// of the wave in device setting. It is called on initialisation and when the selection
// is changed by the user.

void setup::PopulateCombo2() 
{
	WAVEINCAPS wic;
	CString strCurr;
	int i;

	CWnd* pWnd = GetDlgItem(IDC_COMBO1);
	ASSERT_VALID(pWnd);
	LRESULT lCurrent = pWnd->SendMessage(CB_GETCURSEL, 0, 0);
	ASSERT( lCurrent == m_uWaveInDev );

	if( hAutomateDevice != INVALID_HANDLE_VALUE )
	{
		CloseHandle( hAutomateDevice );
		hAutomateDevice = INVALID_HANDLE_VALUE;
	}

	pWnd = GetDlgItem(IDC_COMBO2);
	ASSERT_VALID(pWnd);

	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);

	if( lCurrent == CB_ERR )
		return;

	if( waveInGetDevCaps( lCurrent, &wic, (UINT)sizeof(WAVEINCAPS)) == MMSYSERR_NOERROR )
	{
		for( i=0; Formats[i].formatid != 0; i++ )
		{
			if( Formats[i].formatid == -1 )
			{
				nSampleRate = Formats[i].nSampleRate;
				nBitsPerSample = Formats[i].nBitsPerSample;
				bIsStereo = Formats[i].bIsStereo;
				set_wfx();
				if( IsAudioDeviceValid() )
				{
					strCurr.LoadString(Formats[i].nName);
					pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)strCurr));
				}
			}
			else if( Formats[i].formatid & wic.dwFormats )
			{
				strCurr.LoadString(Formats[i].nName);
				pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)strCurr));
			}
		}
	}

	if( m_uWaveInFormat < 0 )
		m_uWaveInFormat = 0;
	else if( m_uWaveInFormat >= i )
		m_uWaveInFormat = i - 1;
	strCurr.LoadString(Formats[m_uWaveInFormat].nName);
	i = pWnd->SendMessage(CB_FINDSTRING, -1, (LPARAM)(LPCTSTR(strCurr) ) );
	if( i == CB_ERR )
		i = 0;
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)i, 0);
	// we corrupted these so reset them properly
	nSampleRate = Formats[m_uWaveInFormat].nSampleRate;
	nBitsPerSample = Formats[m_uWaveInFormat].nBitsPerSample;
	bIsStereo = Formats[m_uWaveInFormat].bIsStereo;
	SetFilterRate( nSampleRate );
}

// ********************************************************************************
// User selection of wave in device.
// get new selection then populate the formats drop down.

void setup::OnSelendokWaveInDev() 
{
	CWnd* pWnd = GetDlgItem(IDC_COMBO1);
	ASSERT_VALID(pWnd);

	m_uWaveInDev = pWnd->SendMessage(CB_GETCURSEL, 0, 0);
	PopulateCombo2();
}

// ********************************************************************************
// User selects format of audio (speed, number of bits, mono/stereo) from drop down
// list.
// Sets sample rate, number of bits, stereo flag.

void setup::OnSelendokWaveInFormat() 
{
	CString sTemp;
	char szTemp[128];

	CWnd* pWnd = GetDlgItem(IDC_COMBO2);
	ASSERT_VALID(pWnd);

	int i = pWnd->SendMessage(CB_GETCURSEL, 0, 0);
	i = pWnd->SendMessage(CB_GETLBTEXT, (WPARAM)i, (LPARAM)szTemp);
	ASSERT( i != CB_ERR );
	for( i=0; Formats[i].formatid != 0; i++)
	{
		sTemp.LoadString( Formats[i].nName );
		if( strcmp( sTemp, szTemp) == 0 )
			break;
	}
	ASSERT(Formats[i].formatid != 0); 
	if( Formats[i].formatid == 0 )
		i -= 1;
	m_uWaveInFormat = i;
	nSampleRate = Formats[m_uWaveInFormat].nSampleRate;
	nBitsPerSample = Formats[m_uWaveInFormat].nBitsPerSample;
	bIsStereo = Formats[m_uWaveInFormat].bIsStereo;
	SetFilterRate( nSampleRate );
}

// ********************************************************************************
// Save user selections in registry.
// Note that audio sample interval is not yet saved.
// We do not check error returns from write operations - we just assume it is OK.
// TODO - add error handling

void setup::OnButtonSetupSave() 
{
	WAVEINCAPS wic;
	CString sSetup, sToken;
	sSetup.LoadString(IDS_STRING_SETUP);
	CException ex;
	bool bError = false;

	sToken.LoadString(IDS_WAVE_IN_DEVICE);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, m_uWaveInDev ))	
		bError = true;
	sToken.LoadString(IDS_WAVE_IN_FORMAT);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, m_uWaveInFormat ))
		bError = true;
	sToken.LoadString(IDS_ENR_CAL_FILE);
	if( !AfxGetApp( )->WriteProfileString( sSetup, sToken, m_Filename ))
		bError = true;
	sToken.LoadString(IDS_LOG_FILE);
	if( !AfxGetApp( )->WriteProfileString( sSetup, sToken, m_LogFilename ))
		bError = true;
	sToken.LoadString(IDS_AUTOMATE_NAME);
	if( !AfxGetApp( )->WriteProfileString( sSetup, sToken, szAutomateName ))
		bError = true;
	sToken.LoadString(IDS_AUTOMATE_NOISE_SELECTION);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, nAutomateNoiseSelection ))
		bError = true;
	sToken.LoadString(IDS_AUTOMATE_DUT_SELECTION);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, nAutomateDutSelection ))
		bError = true;
	sToken.LoadString(IDS_USE_ENR_FILE);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, bUseEnrFile ))
		bError = true;
	sToken.LoadString(IDS_STRING_USE_HIGH_PASS_FILTER);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, bUseHighPassFilter ))
		bError = true;
	sToken.LoadString(IDS_AUTOMATE_DECIMALS);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, nAutomateDecimalPoints ))
		bError = true;

	sToken.LoadString(IDS_USB_DEVICE);
	if( !AfxGetApp( )->WriteProfileString( sSetup, sToken, m_UsbDeviceName ))
		bError = true;

	sToken.LoadString(IDS_WAVE_IN_DEVICE_NAME);
	if( waveInGetDevCaps( m_uWaveInDev, &wic, (UINT)sizeof(WAVEINCAPS)) == MMSYSERR_NOERROR )
			if( !AfxGetApp( )->WriteProfileString( sSetup, sToken, (LPCTSTR)wic.szPname ))
				bError = true;

	sToken.LoadString(IDS_FFT_BLOCK_SIZE);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, nFFTBlockSize ))
		bError = true;
	if( bError )
		ex.ReportError( MB_OK, IDS_CANNOT_WRITE_REGISTRY );
}

// ********************************************************************************
// This allows selection of the audio sample time.
// Not elegant - we hard code population of control with sequence of entries.

void setup::OnSelendokComboTime() 
{
	CWnd* pWnd = GetDlgItem(IDC_COMBO_TIME);
	ASSERT_VALID(pWnd);

	int i = pWnd->SendMessage(CB_GETCURSEL, 0, 0);
	switch( i )
	{
		case 0:
			nMilliSeconds = 10000;
			break;
		case 2:
			nMilliSeconds = 100;
			break;
		default:
			nMilliSeconds = 1000;
	}
}

// ********************************************************************************
// This is the handler for when the user wants to browse for an ENR calibration file
// Uses common control.
// It does no checking on the file, that is done when the user tries to use it via
// the checkbox (see OnCheckUseEnrFile() )

void setup::OnButtonFile() 
{
    if (!UpdateData(TRUE))
        return;

    CFileDialog dlg1( TRUE,"dat",NULL,
	                 OFN_HIDEREADONLY|OFN_NONETWORKBUTTON|OFN_FILEMUSTEXIST,
	                 _T("ENR Data File (*.dat)|*.dat|Any File Extension|*.*||"));
	
	if( dlg1.DoModal()!=IDOK )
		return;

   	m_Filename=dlg1.GetPathName();

	CWnd* pWnd = GetDlgItem(IDC_EDIT_CONFIG_FILENAME);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText( m_Filename );

	pWnd = GetDlgItem(IDC_CHECK_USE_ENR_FILE);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(BM_SETCHECK, BST_UNCHECKED, 0);
	bUseEnrFile = FALSE;

	UpdateData(FALSE);	// Set the edit box to the filename
}

// ********************************************************************************
// Don't want to check the filename until finished typing, so read & store when 
// loses focus.

void setup::OnKillfocusEditConfigFilename() 
{
	CWnd* pWnd = GetDlgItem(IDC_EDIT_CONFIG_FILENAME);
	ASSERT_VALID(pWnd);
	char szTemp[FILENAME_MAX];

	pWnd->GetWindowText(szTemp, sizeof(szTemp) );
	m_Filename = szTemp;

	m_Filename.TrimLeft();
	m_Filename.TrimRight();
	pWnd->SetWindowText( m_Filename );


	pWnd = GetDlgItem(IDC_CHECK_USE_ENR_FILE);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(BM_SETCHECK, BST_UNCHECKED, 0);
	bUseEnrFile = FALSE;
}

// ********************************************************************************
// When the user chooses a new FFT block size......

void setup::OnSelendokComboFftBlock() 
{
	char szTemp[128];

	CWnd* pWnd = GetDlgItem(IDC_COMBO_FFT_BLOCK);
	ASSERT_VALID(pWnd);

	int i = pWnd->SendMessage(CB_GETCURSEL, 0, 0);
	pWnd->SendMessage(CB_GETLBTEXT, i, (WPARAM)szTemp );

	nFFTBlockSize = atoi( szTemp );
}

// ********************************************************************************
// Handler for when user makes a new selection of device to use for automation.

void setup::OnSelendokComboAuto() 
{
	PopulateComboNoiseDUT();
}

// ********************************************************************************
// Handler for user selecting to use ENR file.
// Tries to read in the ENR file. If valid, uses it. 
// If not gives error and turns off again


void setup::OnCheckUseEnrFile() 
{
	CException ex;
	int i;
		
	CWnd* pWnd = GetDlgItem(IDC_CHECK_USE_ENR_FILE);
	ASSERT_VALID(pWnd);

	bUseEnrFile = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
	if( bUseEnrFile )
	{
		if( (i = ReadValidEnrFile() ) != 0 )
		{	ex.ReportError( MB_OK, i );
			bUseEnrFile = FALSE;
			pWnd->SendMessage(BM_SETCHECK, BST_UNCHECKED, 0);
		}
	}
}

// ********************************************************************************
// This function readin in the current calibration file, parses it 
// and populates the ENR calibration table.
//
// It returns an error value in case of failure and returns zero if all OK.
// Error values are string indexes from the string table.
// The global variable CalibrationTableSize is set or cleared appropriately

int setup::ReadValidEnrFile()
{

	FILE *fp;
	char szTemp[256];
	int nRowCount = 0;
	char *token;
	double dFreq, dEnr;

	nCalibrationTableSize = 0;
	if( strlen( m_Filename ) == 0 )
		return IDS_CAL_NO_FILE_GIVEN;
	fp = fopen( m_Filename, "r" );
	if( fp == NULL )
		return IDS_CAL_FILE_NO_EXIST;
	while( !feof( fp ))
	{
		if( fgets( szTemp, sizeof(szTemp)-1, fp) != NULL )
		{
			int i = strlen( szTemp );
			if( i == 0 )
			{
				if( ferror( fp ) )
				{ fclose( fp ); return IDS_FILE_READ_ERROR; }
			}
			else
			{
				if( i >= sizeof(szTemp)-2 )
				{ fclose( fp ); return IDS_LINE_TOO_LONG; }
				if( i == 1 || szTemp[0] == ';' || szTemp[0] == '#' || szTemp[0] == '/' || szTemp[0] == '\\' )
					continue;
				token = strtok( szTemp, ",\t;\n" );
				if(strlen( token ) == 0 )
				{ fclose( fp ); return IDS_MISSING_FREQ; }
				dFreq = atof( token );
				if( dFreq < 0 || dFreq > MAX_PERMITTED_FREQ_MHZ)
				{ fclose( fp ); return IDS_FREQ_ERROR_RANGE; }
				token = strtok( NULL, ",\t;\n"  );
				if(strlen( token ) == 0 )
				{ fclose( fp ); return IDS_MISSING_ENR; }
				dEnr = atof( token );
				if( dEnr < 0 || dEnr > MAX_PERMITTED_ENR_DB)
				{ fclose( fp ); return IDS_ENR_ERROR_RANGE; }
				CalibrationTable[nRowCount].dFreq = dFreq;
				CalibrationTable[nRowCount].dEnr = dEnr;
				nRowCount++;
				if( nRowCount >= CAL_TABLE_SIZE )
				{ fclose( fp ); return IDS_FILE_TOO_BIG; }
			}
		}
		else if( ferror( fp ) )
			{ fclose( fp ); return IDS_FILE_READ_ERROR; }
	}
	CalibrationTable[nRowCount].dFreq = CalibrationTable[nRowCount].dEnr = (double)0;
	fclose( fp );
	if( nRowCount == 0 )
		return IDS_EMPTY_CAL_TABLE;
	qsort( CalibrationTable, nRowCount, sizeof(cal_table), CalibrationTableSortCompare );
	nCalibrationTableSize = nRowCount;
	return 0;
}

// ********************************************************************************
// Simple helper to above - a compare function called by qsort().

int CalibrationTableSortCompare( const void *arg1, const void *arg2 )
{
	double dTemp = (( cal_table *) arg1)->dFreq - (( cal_table* )arg2)->dFreq;
	if( dTemp < 0 ) return -1;
	else if( dTemp == 0 ) return 0;
	else return 1;
}

// ********************************************************************************
// Function called externally. Given a frequency it uses the ENR calibration file
// to return an ENR value. If the ENR file is not in use it just returns the current
// ENR value
//
// If frequency <= first entry in table, use that ENR value
// If frequency >= last  entry in table, use that ENR value
// If in between two entries, linearly interpolate.

double setup::ComputeENR( double dEnrOrFrequency )
{
	int i;

	if( dEnrOrFrequency < 0 ) 
		dEnrOrFrequency = 0;
	if( ! bUseEnrFile )
		return dEnrOrFrequency;
	if( dEnrOrFrequency <= CalibrationTable[0].dFreq )
		return CalibrationTable[0].dEnr;
	if( dEnrOrFrequency >= CalibrationTable[nCalibrationTableSize-1].dFreq )
		return CalibrationTable[nCalibrationTableSize-1].dEnr;
	for( i=0; i< nCalibrationTableSize; i++ )
		if( dEnrOrFrequency <= CalibrationTable[i].dFreq )
			break;
	i--;
	return (dEnrOrFrequency-CalibrationTable[i].dFreq) / 
			(CalibrationTable[i+1].dFreq - CalibrationTable[i].dFreq) *
			(CalibrationTable[i+1].dEnr - CalibrationTable[i].dEnr) +
			CalibrationTable[i].dEnr;
}

// ********************************************************************************
// Called by framework on valid user selection of noise source switch selection
// Currently only handles COM port. It reads the selection and sets the DUT option
// Needs LPT adding.

void setup::OnSelendokComboNoise() 
{
	CWnd *pWnd = GetDlgItem(IDC_COMBO_NOISE);
	ASSERT_VALID(pWnd);
	nAutomateNoiseSelection = pWnd->SendMessage(CB_GETCURSEL, 0, 0);
	if( strncmp( szAutomateName, "COM", 3 ) == 0 )
	{
		nAutomateDutSelection = nAutomateNoiseSelection == 0 ? 1 : 0;
		pWnd = GetDlgItem(IDC_COMBO_DUT);
		ASSERT_VALID(pWnd);
		pWnd->SendMessage(CB_SETCURSEL, (WPARAM)nAutomateDutSelection, 0);
	}

	// to be done

	else if( strncmp( szAutomateName, "USB", 3 ) == 0 )
	{
		// whilst we could set the bit to one other than noise selection
		// it is getting a bit complex to manage so leave it
	}
}

// ********************************************************************************
// Function called by measurement routines. Switches the noise and DUT controls.
// Two inputs, switch ON if TRUE for each.
// Currently only handles COM port.

void setup::SwitchDevices( bool bNoise, bool bDut )
{
	int nCommsSetting;
	CException ex;

	if( strncmp( szAutomateName, "COM", 3 ) == 0 )
	{
		if( hAutomateDevice == INVALID_HANDLE_VALUE )
			hAutomateDevice = CreateFile(
								szAutomateName, 
								GENERIC_READ | GENERIC_WRITE, // desied access
								0,					// No sharing
								NULL,				// security attributes
								OPEN_EXISTING,		// disposition
								FILE_ATTRIBUTE_NORMAL, // flags and attributes
								NULL );				// template file
		if( hAutomateDevice == INVALID_HANDLE_VALUE )
		{
			int i = GetLastError();
			ex.ReportError( MB_OK, IDS_CANNOT_OPEN_COMPORT );

			// TODO : do something better with the error (like display it)
		}
		else
		{
			if( nAutomateNoiseSelection == 0 )
				nCommsSetting = bNoise ? SETRTS : CLRRTS;
			else
				nCommsSetting = bNoise ? SETDTR : CLRDTR;
			if( !EscapeCommFunction( hAutomateDevice, nCommsSetting ))
			{
				ex.ReportError( MB_OK, IDS_CANNOT_SET_COMPORT );
				return;
			}

			if( nAutomateDutSelection == 0 )
				nCommsSetting = bDut ? SETRTS : CLRRTS;
			else
				nCommsSetting = bDut ? SETDTR : CLRDTR;
			if( !EscapeCommFunction( hAutomateDevice, nCommsSetting ))
				ex.ReportError( MB_OK, IDS_CANNOT_SET_COMPORT );
		}
	}

	if( strncmp( szAutomateName, "USB", 3 ) == 0 )
	{
		VNA_TXBUFFER tx_data;
		tx_data.raw.portD = 0;
		if( ! bNoise)
			tx_data.raw.portD |= unsigned char(1 << nAutomateNoiseSelection);
		if( ! bDut)
			tx_data.raw.portD |= unsigned char(1 << nAutomateDutSelection);
		tx_data.raw.command_code = 0x5A;
		tx_data.raw.flags = CmdVnaRawDataFlagsWriteD;
		if( vnawrite( &tx_data, sizeof(tx_data.raw) ) == false )
		{
			ex.ReportError( MB_OK, IDS_CANNOT_WRITE_USBPORT );
		}
	}

	return;
}

// ********************************************************************************
// when user clicks button to browse for a log file, we get called.
// uses common control to select file and stores filename

void setup::OnButtonLogFile() 
{
    if (!UpdateData(TRUE))
        return;

    CFileDialog dlg1( TRUE,"log",NULL,
	                 OFN_HIDEREADONLY|OFN_NONETWORKBUTTON|OFN_FILEMUSTEXIST,
	                 _T("Log File (*.log)|*.log|Any File Extension|*.*||"));
	
	if( dlg1.DoModal()!=IDOK )
		return;

   	m_LogFilename=dlg1.GetPathName();

	CWnd* pWnd = GetDlgItem(IDC_EDIT_LOG_FILENAME);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText( m_LogFilename );

	UpdateData(FALSE);	// Set the edit box to the filename
}

// ********************************************************************************
// when they finish editing the filename, store it

void setup::OnKillfocusEditLogFilename() 
{
	CWnd* pWnd = GetDlgItem(IDC_EDIT_LOG_FILENAME);
	ASSERT_VALID(pWnd);
	char szTemp[FILENAME_MAX];

	pWnd->GetWindowText(szTemp, sizeof(szTemp) );
	m_LogFilename = szTemp;
	m_LogFilename.TrimLeft();
	m_LogFilename.TrimRight();
	pWnd->SetWindowText( m_LogFilename );

}

// ********************************************************************************
// when tick or untick flag to log results

void setup::OnCheckLogResults() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_LOG_RESULTS);
	ASSERT_VALID(pWnd);

	bUseLogFile = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;

	if( bUseLogFile && (m_LogFilename.GetLength() == 0) )
	{
		bUseLogFile = FALSE;
		pWnd->SendMessage(BM_SETCHECK, BST_UNCHECKED, 0);
	}
}

// ********************************************************************************
// manage selection of high pass filter

void setup::OnCheckHighPass() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_HIGH_PASS);
	ASSERT_VALID(pWnd);

	bUseHighPassFilter = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
}

// ********************************************************************************
// Manage selection of device option for DUT switching

void setup::OnSelendokComboDut() 
{
	CWnd *pWnd = GetDlgItem(IDC_COMBO_DUT);
	ASSERT_VALID(pWnd);
	if( strncmp( szAutomateName, "COM", 3 ) == 0 )
	{
	}
	else if( strncmp( szAutomateName, "USB", 3 ) == 0 )
	{
		nAutomateDutSelection = pWnd->SendMessage(CB_GETCURSEL, 0, 0);
	}
}
