// measure.cpp : implementation file
//
//(C) Copyright Dave Roberts G8KBB 2005
// This program is released only for the purpose of self training and eduation
// in amatuer radio. It may not be used for any commercial purpose, sold or
// modified.
// It has been written as an exercise in self tuition by me as part of my hobby
// and I make no claims about its fitness for purpose or correct operation.

#include "stdafx.h"
#include "NoiseMeter.h"
#include "measure.h"

#include "mmsystem.h"
#include "spectrum.h"
#include "setup.h"

#include "common.h"

#include "high-pass-filters.h"

#include "math.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

extern spectrum *pSpectrum;
extern measure *pMeasure;
extern setup *pSetup;

extern CString sLogResults;

void CALLBACK waveGotBlockMeasure(
  HWAVEIN hwi,       
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2     
);

int m_bFontSize;


/////////////////////////////////////////////////////////////////////////////
// measure property page

IMPLEMENT_DYNCREATE(measure, CPropertyPage)

measure::measure() : CPropertyPage(measure::IDD)
{
	//{{AFX_DATA_INIT(measure)
	m_bDoAverage = FALSE;
	//}}AFX_DATA_INIT
	CString sSetup, sToken;
	sSetup.LoadString(IDS_STRING_SETUP);

	sToken.LoadString(IDS_TRACE_AVERAGE_COUNT);
	m_nAverageCount = AfxGetApp( )->GetProfileInt( sSetup, sToken, 1 );
	sToken.LoadString(IDS_TRACE_AVERAGE_FLAG);
	m_bDoAverage = AfxGetApp( )->GetProfileInt( sSetup, sToken, 0 );
	sToken.LoadString(IDS_FONT_SIZE);
	m_bFontSize = AfxGetApp( )->GetProfileInt( sSetup, sToken, 24 );
}

measure::~measure()
{
}

void measure::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(measure)
	DDX_Check(pDX, IDC_CHECK_AVERAGE, m_bDoAverage);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(measure, CPropertyPage)
	//{{AFX_MSG_MAP(measure)
	ON_BN_CLICKED(IDC_CHECK_AVERAGE, OnCheckAverage)
	ON_CBN_SELENDOK(IDC_COMBO_AVG_COUNT, OnSelendokComboAvgCount)
	ON_BN_CLICKED(IDC_BUTTON_MEASURE_SAVE, OnButtonMeasureSave)
	ON_BN_CLICKED(IDC_BUTTON_STORE, OnButtonStoreReading)
	ON_BN_CLICKED(IDC_CHECK_NOISE_ON, OnCheckNoiseOn)
	ON_BN_CLICKED(IDC_CHECK_DUT_ON, OnCheckDutOn)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// measure message handlers

bool bShowDifferenceInReading;
double dPower[MAX_AVERAGE];
bool bWrappedBuffer;
double dLastDisplayReading;
double dCompareReading;
double dCrest;
CFont fnMyFont24;

bool bNoiseSouceOn;
bool bDutInCircuit;

BOOL measure::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	char szTemp[16];
	
	CWnd* pWnd = GetDlgItem(IDC_COMBO_AVG_COUNT);
	ASSERT_VALID(pWnd);

	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	for( UINT uCount =1; uCount < MAX_AVERAGE; uCount++)
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)_itoa(uCount,szTemp,10)));
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)m_nAverageCount-1, 0);

	bShowDifferenceInReading = FALSE;
	bRunning = FALSE;
	bNoiseSouceOn = FALSE;
	bDutInCircuit = FALSE;

	fnMyFont24.CreateFont( m_bFontSize, 0, 0, 0, FW_MEDIUM, 0, 0, 0, ANSI_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, NULL );
//	fnMyFont20.CreateFont( 20, 0, 0, 0, FW_MEDIUM, 0, 0, 0, ANSI_CHARSET, OUT_CHARACTER_PRECIS, OUT_CHARACTER_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, NULL );

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}

// ********************************************************************************
// when we are active, do this.
// Sort out display and start audio functions

BOOL measure::OnSetActive() 
{
	CException ex;
	CWnd* pWnd;

	pWnd = GetDlgItem(IDC_COMBO_AVG_COUNT);
	ASSERT_VALID(pWnd);
	int i = pWnd->SendMessage(CB_SETCURSEL, (WPARAM)m_nAverageCount-1, 0);

	SET_STATE( IDC_CHECK_NOISE_ON, pSetup->bEnableAutomation );
	SET_STATE( IDC_CHECK_DUT_ON, pSetup->bEnableAutomation );
	SET_STATE( IDC_STATIC_M_SWITCH, pSetup->bEnableAutomation );

	if( pSetup->bEnableAutomation )
		pSetup->SwitchDevices( bNoiseSouceOn, bDutInCircuit );
	set_wfx();

	ASSERT( !bRunning);
	bWrappedBuffer = FALSE;
	if( IsAudioDeviceValid() )
		if( StartReading((DWORD)waveGotBlockMeasure)!= MMSYSERR_NOERROR )
				ex.ReportError( MB_OK, IDS_CANNOT_OPEN_WAVEIN );
	return CPropertyPage::OnSetActive();
}

// ********************************************************************************
// This is the WaveInProc function for the measure process.
// It calculates readings and displays them as needed.

void CALLBACK waveGotBlockMeasure(
  HWAVEIN hwi,       
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2 )    
{
	// Only handle data messages when audio system running

	if( (uMsg == WIM_DATA) && (bRunning == TRUE))
	{
		__int64 u64sum = 0;
		double dAverage;
		double dResult;
		SHORT *ptr = (short *)(((LPWAVEHDR)dwParam1)->lpData);
		unsigned char *ptrc = (unsigned char *)(((LPWAVEHDR)dwParam1)->lpData);
		DWORD i,j;
		int nMax = 0;
		double dMax;
		double dMean;
		int uMean = 0;
		__int32 x;
		__int16 y;

		int nAdd2ForStereo = wfx.nChannels == 1 ? 1 : 2;

		sLogResults = "";

		DWORD dwBytesRecorded = ((LPWAVEHDR)dwParam1)->dwBytesRecorded;
		switch( wfx.wBitsPerSample )
		{
			case 24:
				j = 3 * nAdd2ForStereo;
				FilterInit( ptrc[0] + (ptrc[1]<<8) + (ptrc[2]<<16) +( ptrc[2] &0x80? 255<<24: 0 ) );
				for( i = 0; i < dwBytesRecorded; i+= j, ptrc += j)
				{
					x = ptrc[0] + (ptrc[1]<<8) + (ptrc[2]<<16) +( ptrc[2] &0x80? 255<<24: 0 );
					if( pSetup->bUseHighPassFilter )
						x = (__int32)((*pFilter)( x ));
					if( abs( x ) > nMax )
						nMax = x;
					u64sum += (__int64(x) * (x));
					uMean += x;
				}
				dMean = (double)uMean / (dwBytesRecorded/j);
				dMax = ((double)nMax -dMean)/ (32768*256);
				if( dMean < 0 ) 
					dMean = -dMean;
				dResult = (double)u64sum / (dwBytesRecorded/j);
				dResult = (sqrt( dResult ) - dMean)/(32768*256);
				break;

			case 16:
				j = 2 * nAdd2ForStereo;
				FilterInit( *ptr );
				for( i = 0; i < dwBytesRecorded; i+= j, ptr += nAdd2ForStereo)
				{
					y = *ptr;
					if( pSetup->bUseHighPassFilter )
						y = (__int16)((*pFilter)( y ));
					if( abs( y ) > nMax )
						nMax = y;
					u64sum += (__int32(y) * (y));
					uMean += *ptr;
				}
				dMean = (double)uMean / (dwBytesRecorded/j);
				dMax = ((double)nMax -dMean)/ 32768;
				if( dMean < 0 ) 
					dMean = -dMean;
				dResult = (double)u64sum / (dwBytesRecorded/j);
				dResult = (sqrt( dResult ) - dMean)/32768;
				break;
			case 8:
				j = nAdd2ForStereo;
				FilterInit( *ptrc );
				for( i = 0; i < dwBytesRecorded; i += j, ptrc += nAdd2ForStereo)
				{
					y = *ptrc;
					if( pSetup->bUseHighPassFilter )
						y = (__int16)((*pFilter)( y ));
					if( y > nMax )
						nMax = y;
					u64sum += (__int16(y) * (y));
					uMean += y;
				}
				dMean = (double)uMean / (dwBytesRecorded/j);
				dMax = ((double)nMax - dMean)/128;
				if( dMean < 0 ) 
					dMean = -dMean;
				dResult = (double)u64sum / (dwBytesRecorded/j);
				dResult = (sqrt( dResult ) - dMean)/128;
				break;
			default:
				ASSERT( FALSE );
		}
		if( dResult < 0 ) 
			dResult = 0-dResult;
		dCrest = dMax / dResult;
		if(dResult < (double)0.000001)
			dResult = -120;
		else
			dResult = 20* log10( dResult);
		dPower[nDataStoreIndex] = dResult;
		int nAverageCount = pMeasure->m_nAverageCount;
		if( nAverageCount < 0 )
			nAverageCount = 0;
		if( nAverageCount > MAX_AVERAGE )
			nAverageCount  = MAX_AVERAGE;
		if( pMeasure->m_bDoAverage )
		{
			int nIndex = nDataStoreIndex;
			int nCount = nAverageCount;
			if( bWrappedBuffer == FALSE )
				nCount = nAverageCount > nDataStoreIndex ? nDataStoreIndex : nAverageCount;
			dAverage = 0;
			int nLoops = nCount;
			while( nLoops-- )
			{
				dAverage += dPower[nIndex];
				if( nIndex == 0 )
					nIndex = MAX_AVERAGE-1;
				else
					nIndex--;
			}
			dAverage /= nCount;
		}
		dLastDisplayReading = pMeasure->m_bDoAverage ? dAverage : dResult;
		if( ++nDataStoreIndex >= MAX_AVERAGE )
		{
			nDataStoreIndex = 0;
			bWrappedBuffer = TRUE;
		}

		CWnd* pWnd = pMeasure->GetDlgItem(IDC_EDIT_SHOW_DB);
		ShowReading( pWnd, dResult==0, dResult, IDS_STRING_WAIT, &fnMyFont24, TRUE);
		if( pMeasure->m_bDoAverage )
		{
			CWnd* pWnd = pMeasure->GetDlgItem(IDC_EDIT_SHOW_DB_AVERAGED);
			ShowReading( pWnd, dAverage==0, dAverage, IDS_STRING_WAIT, &fnMyFont24, TRUE);
		}
		if( bShowDifferenceInReading )
		{
			pWnd = pMeasure->GetDlgItem(IDC_EDIT_DB_RELATIVE);
			ShowReading( pWnd, dLastDisplayReading==0, dLastDisplayReading- dCompareReading, IDS_STRING_WAIT, &fnMyFont24);
		}
		pWnd = pMeasure->GetDlgItem(IDC_EDIT_CREST);
		ShowReading( pWnd, dCrest < 0 || dCrest > 100, dCrest, IDS_STRING_ERROR, &fnMyFont24, TRUE);

		if( pSetup->bUseLogFile )
		{
			FILE *fp;
			CException ex;

			if( (fp = fopen( pSetup->m_LogFilename, "a" )) != NULL )
			{
				CTime theTime = CTime::GetCurrentTime();
				CString t = theTime.Format("%c,");
				t += bNoiseSouceOn ? "Noise On," : "Noise Off,";
				t += bDutInCircuit ? "DUT in Circuit," : "DUT out of Circuit,";
				fprintf(fp,"Measure,%s %s\n", (LPCTSTR)t,(LPCTSTR)sLogResults );
				fclose(fp);
			}
			else
				ex.ReportError( MB_OK, IDS_CANNOT_OPEN_LOGFILE );
		}
		MMRESULT mmResult = waveInAddBuffer( hwi, (LPWAVEHDR)dwParam1, sizeof( WAVEHDR) );
	}
}

// ********************************************************************************
// User sets or clears the flag to average readings.

void measure::OnCheckAverage() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_AVERAGE);
	ASSERT_VALID(pWnd);

	m_bDoAverage = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;

	if( !m_bDoAverage )
	{
		pWnd = GetDlgItem(IDC_EDIT_SHOW_DB_AVERAGED);
		ASSERT_VALID(pWnd);
		pWnd->SetWindowText( "" );
	}
}

// ********************************************************************************
// User changes the number of readings to average.

void measure::OnSelendokComboAvgCount() 
{
	CWnd* pWnd = GetDlgItem(IDC_COMBO_AVG_COUNT);
	ASSERT_VALID(pWnd);

	m_nAverageCount = pWnd->SendMessage(CB_GETCURSEL , 0, 0) + 1;
	
}

// ********************************************************************************
// User wants to save current settings. From this screen we save the average count
// and average flag.

void measure::OnButtonMeasureSave() 
{
	CString sSetup, sToken;
	sSetup.LoadString(IDS_STRING_SETUP);
	CException ex;
	bool bError = false;

	sToken.LoadString(IDS_TRACE_AVERAGE_COUNT);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, m_nAverageCount ))
		bError = true;	
	sToken.LoadString(IDS_TRACE_AVERAGE_FLAG);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, m_bDoAverage ))
		bError = true;	
	sToken.LoadString(IDS_FONT_SIZE);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, m_bFontSize ))
		bError = true;
	if( bError )
		ex.ReportError( MB_OK, IDS_CANNOT_WRITE_REGISTRY );
}

// ********************************************************************************
// User pressed the store reading function. Saves current reading and sets flags
// to enable display of relaive reading

void measure::OnButtonStoreReading() 
{
	bShowDifferenceInReading = TRUE;
	dCompareReading = dLastDisplayReading;
	
}

// ********************************************************************************
// When we move away from this page, stop audio functions

BOOL measure::OnKillActive() 
{
	StopInput();

	return CPropertyPage::OnKillActive();
}

// ********************************************************************************
// If the user changes the Noise switch setting, we get called. Read status then call
// switch function.

void measure::OnCheckNoiseOn() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_NOISE_ON);
	ASSERT_VALID(pWnd);

	bNoiseSouceOn = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
	pSetup->SwitchDevices( bNoiseSouceOn, bDutInCircuit );
}

// ********************************************************************************
// If the user changes the DUT switch setting, we get called. Read status then call
// switch function.

void measure::OnCheckDutOn() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_DUT_ON);
	ASSERT_VALID(pWnd);

	bDutInCircuit = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
	pSetup->SwitchDevices( bNoiseSouceOn, bDutInCircuit );
}

