// deviceNF.cpp : implementation file
//
//(C) Copyright Dave Roberts G8KBB 2005
// This program is released only for the purpose of self training and eduation
// in amatuer radio. It may not be used for any commercial purpose, sold or
// modified.
// It has been written as an exercise in self tuition by me as part of my hobby
// and I make no claims about its fitness for purpose or correct operation.


#include "stdafx.h"
#include "NoiseMeter.h"
#include "deviceNF.h"
#include "measure.h"

#include "mmsystem.h"

#include "spectrum.h"
#include "setup.h"

#include "common.h"

#include "math.h"

#include "WaitingForMeasurement.h"

#include "high-pass-filters.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// WaveInProc used to handle buffers returned with audio data
//
void CALLBACK waveGotBlockNF(
  HWAVEIN hwi,       
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2     
);

// Set of structures needed for handling waveform input
// but we only need access to wfx for the waveInProc()
//extern HWAVEIN hwi;
extern WAVEFORMATEX wfx;
//extern WAVEHDR wh1;

// storage for computed power readings
double dPowerNF[MAX_AVERAGE];

// Pointers to other objects
extern spectrum *pSpectrum;
extern measure *pMeasure;
extern setup *pSetup;
extern deviceNF *pDeviceNF;

// fonts used other than system font in the displays
// no longer ise font size 20
//extern CFont fnMyFont20;
extern CFont fnMyFont24;

// storage for 4 measurements made for noise figure measurement
// In sequence these are reference on, referenc off, DUT on & DUT off
double dAverageNF[4];
// boolean flags showing which of the above readings are valid
bool bGotReadingNF[4];
// If set, measure overall noise figure not that of DUT.
// In other words measure F2 alone not F12
bool bMeasureOverallNoiseFigure;

bool i_am_done;
int nResponseCode;

// How many dummy reads before valid reads to allow settling time?
int uDummyReads;

// string used in logging to create calculated results
CString sLogResults;

// string used to put reading name into dialog 
CString sReadingName;
// and index for strings used
int ReadingNames[] =
{
	IDS_STRING_NON,
	IDS_STRING_NOFF,
	IDS_STRING_DON,
	IDS_STRING_DOFF
};



// Dialog shown whilst reading - we want to point to it to update text on it
WaitingForMeasurement dlgWaitMessage;

/////////////////////////////////////////////////////////////////////////////
// deviceNF property page

IMPLEMENT_DYNCREATE(deviceNF, CPropertyPage)

deviceNF::deviceNF() : CPropertyPage(deviceNF::IDD)
{
	//{{AFX_DATA_INIT(deviceNF)
	//}}AFX_DATA_INIT

	CString sSetup, sToken;
	sSetup.LoadString(IDS_STRING_SETUP);

	sToken.LoadString(IDS_DUMMY_READS);
	m_uDummyReads = AfxGetApp( )->GetProfileInt( sSetup, sToken, 1 );

	sToken.LoadString(IDS_ENR);
	CString szTemp = AfxGetApp( )->GetProfileString( sSetup, sToken, DEFAULT_ENR );
	dEnrOrFrequency = atof( szTemp );
	dEnr = 0; // program computes this later

	sToken.LoadString(IDS_TSOFF);
	szTemp = AfxGetApp( )->GetProfileString( sSetup, sToken, DEFAULT_TSOFF );
	dTSOff = atof( szTemp );

	sToken.LoadString(IDS_T0);
	szTemp = AfxGetApp( )->GetProfileString( sSetup, sToken, DEFAULT_T0 );
	dT0 = atof( szTemp );

	sToken.LoadString(IDS_AUTOMATE_NF_NOISE);
	m_bNoiseSwitchAuto = AfxGetApp( )->GetProfileInt( sSetup, sToken, FALSE ) & 1;

	sToken.LoadString(IDS_AUTOMATE_NF_DUT);
	m_bDutSwitchAuto = AfxGetApp( )->GetProfileInt( sSetup, sToken, FALSE ) & 1;

	bMeasureRepeat = FALSE;
	bMeasureOverallNoiseFigure = FALSE;
}


deviceNF::~deviceNF()
{
	char szTemp[128];
	CString sSetup, sToken;
	sSetup.LoadString(IDS_STRING_SETUP);

	sToken.LoadString(IDS_ENR);
	sprintf( szTemp, "%.3f", dEnrOrFrequency );
	int j = AfxGetApp( )->WriteProfileString( sSetup, sToken, szTemp );

	sToken.LoadString(IDS_TSOFF);
	sprintf( szTemp, "%.2f", dTSOff );
	j = AfxGetApp( )->WriteProfileString( sSetup, sToken, szTemp );

	sToken.LoadString(IDS_T0);
	sprintf( szTemp, "%.2f", dT0 );
	j = AfxGetApp( )->WriteProfileString( sSetup, sToken, szTemp );
}

void deviceNF::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(deviceNF)
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(deviceNF, CPropertyPage)
	//{{AFX_MSG_MAP(deviceNF)
	ON_BN_CLICKED(IDC_CHECK_NOISE_SWITCH, OnCheckNoiseSwitch)
	ON_BN_CLICKED(IDC_CHECK_DUT_SWITCH, OnCheckDutSwitch)
	ON_BN_CLICKED(IDC_BUTTON_REF_ON, OnButtonRefOn)
	ON_BN_CLICKED(IDC_BUTTON_REF_OFF, OnButtonRefOff)
	ON_BN_CLICKED(IDC_BUTTON_DEVICE_ON, OnButtonDeviceOn)
	ON_BN_CLICKED(IDC_BUTTON_DEVICE_OFF, OnButtonDeviceOff)
	ON_EN_CHANGE(IDC_EDIT_ENR, OnChangeEditEnr)
	ON_EN_CHANGE(IDC_EDIT_TSOFF, OnChangeEditTsoff)
	ON_BN_CLICKED(IDC_CHECK_AVERAGE, OnCheckAverage)
	ON_CBN_SELENDOK(IDC_COMBO_AVG_COUNT, OnSelendokComboAvgCount)
	ON_BN_CLICKED(IDC_BUTTON_MEASURE_SAVE, OnButtonMeasureSave)
	ON_BN_CLICKED(IDC_CHECK_OVERALL_NF_MODE, OnCheckOverallNfMode)
	ON_CBN_SELENDOK(IDC_COMBO_DUMMIES, OnSelendokComboDummies)
	ON_BN_CLICKED(IDC_CHECK_REPEAT, OnCheckRepeat)
	ON_BN_CLICKED(IDC_BUTTON_CLR, OnButtonClr)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// deviceNF message handlers

BOOL deviceNF::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	int nCount;
	char szTemp[128];

	CWnd* pWnd = GetDlgItem(IDC_COMBO_AVG_COUNT);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	for( nCount =1; nCount < MAX_AVERAGE; nCount++)
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)_itoa(nCount,szTemp,10)));
	pWnd = GetDlgItem(IDC_COMBO_DUMMIES);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	for( nCount =0; nCount <= MAX_DUMMY_READS; nCount++)
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)_itoa(nCount,szTemp,10)));
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)m_uDummyReads, 0);

	bGotReadingNF[0] = bGotReadingNF[1] = bGotReadingNF[2] = bGotReadingNF[3] = FALSE;	

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}

// ********************************************************************************
// WHen the user checks or unchecks the DUT switch box, call this. Reads state an
// then sets button configuration.

void deviceNF::OnCheckDutSwitch() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_DUT_SWITCH);
	ASSERT_VALID(pWnd);

	m_bDutSwitchAuto = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
	SetButtonConfig();
	
}

// ********************************************************************************
// WHen the user checks or unchecks the Noise switch box, call this. Reads state an
// then sets button configuration.

void deviceNF::OnCheckNoiseSwitch() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_NOISE_SWITCH);
	ASSERT_VALID(pWnd);

	m_bNoiseSwitchAuto = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
	SetButtonConfig();
	
}


// ********************************************************************************
// Helper utility to set the configuration of on screen buttons etc.
// enable or disable window controls as needed

void deviceNF::SetButtonConfig()
{
	CWnd *pWnd;
	
	if( pSetup->bEnableAutomation == FALSE)
		m_bNoiseSwitchAuto = FALSE;

	if( pSetup->bEnableAutomation == FALSE)
		m_bDutSwitchAuto = FALSE;

	SET_STATE( IDC_CHECK_NOISE_SWITCH, pSetup->bEnableAutomation );
	SET_STATE( IDC_CHECK_DUT_SWITCH, pSetup->bEnableAutomation && !bMeasureOverallNoiseFigure );
	SET_STATE( IDC_BUTTON_REF_OFF, !m_bNoiseSwitchAuto );
	SET_STATE( IDC_BUTTON_DEVICE_ON, !( bMeasureOverallNoiseFigure || m_bDutSwitchAuto) );
	SET_STATE( IDC_BUTTON_DEVICE_OFF, !( bMeasureOverallNoiseFigure || m_bDutSwitchAuto || m_bNoiseSwitchAuto) );
	SET_STATE( IDC_STATIC_DEVICE, !bMeasureOverallNoiseFigure );
	SET_STATE( IDC_STATIC_DUT_ON, !bMeasureOverallNoiseFigure );
	SET_STATE( IDC_STATIC_DUT_OFF, !bMeasureOverallNoiseFigure );
	SET_STATE( IDC_EDIT_DEVICE_ON, !bMeasureOverallNoiseFigure );
	SET_STATE( IDC_EDIT_DEVICE_OFF, !bMeasureOverallNoiseFigure );
	SET_STATE( IDC_STATIC_GAIN, !bMeasureOverallNoiseFigure );
	SET_STATE( IDC_EDIT_GAIN, !bMeasureOverallNoiseFigure );

	pWnd = GetDlgItem(IDC_STATIC_REFERENCE);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText( bMeasureOverallNoiseFigure ? "System" : "Reference");
}

// ********************************************************************************
// When the property page becomes active, perform final configurations

BOOL deviceNF::OnSetActive() 
{
	CString sTemp;
	char sztemp[128];
	CWnd *pWnd;

	SetButtonConfig();

	pWnd = GetDlgItem(IDC_EDIT_ENR);
	ASSERT_VALID(pWnd);
	sprintf( sztemp, "%.3f", dEnrOrFrequency);
	pWnd->SetWindowText( sztemp );
	pWnd = GetDlgItem(IDC_EDIT_TSOFF);
	ASSERT_VALID(pWnd);
	sprintf( sztemp, "%.2f", dTSOff);
	pWnd->SetWindowText( sztemp );
	pWnd = GetDlgItem(IDC_CHECK_AVERAGE);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(BM_SETCHECK, pMeasure->m_bDoAverage, 0);

	SET_STATE( IDC_STATIC_ENR_FROM_F, pSetup->bUseEnrFile );

	pWnd = GetDlgItem(IDC_STATIC_USE_F);
	ASSERT_VALID(pWnd);
	sTemp.LoadString( pSetup->bUseEnrFile ? IDS_STRINGNF_ENR_F : IDS_STRINGNF_ENR_DB );
	pWnd->SetWindowText(sTemp);

	pWnd = GetDlgItem(IDC_COMBO_AVG_COUNT);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)pMeasure->m_nAverageCount-1, 0);

	pWnd = GetDlgItem(IDC_CHECK_NOISE_SWITCH);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(BM_SETCHECK, m_bNoiseSwitchAuto, 0);

	pWnd = GetDlgItem(IDC_CHECK_DUT_SWITCH);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(BM_SETCHECK, m_bDutSwitchAuto, 0);

	return CPropertyPage::OnSetActive();
}

// ********************************************************************************
// WaveInProc() for the noise figure measurement.
// It is called by the audio subsystem. 

void CALLBACK waveGotBlockNF(
  HWAVEIN hwi,       
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2 )    
{
	// we only handle the WIM_DATA messages and only if audio input is enabled

//	if( !bRunning )
//		bStopped = true;
	if( uMsg == WIM_DATA && bRunning == TRUE )
	{
		char sztemp[128];
		__int64 u64sum = 0;
		double dResult;
		SHORT *ptr = (short *)(((LPWAVEHDR)dwParam1)->lpData);
		unsigned char *ptrc = (unsigned char *)(((LPWAVEHDR)dwParam1)->lpData);
		DWORD i, j;
		int nMax = 0;
		double dMax;
		int nAdd2ForStereo = wfx.nChannels == 1? 1 : 2;
		double dMean;
		int uMean = 0;
		__int32 x;
		__int16 y;

		if( i_am_done)
			return;
		CWnd *pWnd = dlgWaitMessage.GetDlgItem(IDC_STATIC_PAUSE);
		ASSERT_VALID(pWnd);

		// handle dummy reads. If any still to do, then simple add the buffer
		// back in again, uodate the onscreen display and decrement the dummy count
		if( uDummyReads > 0 )
		{
			MMRESULT mmResult = waveInAddBuffer( hwi, (LPWAVEHDR)dwParam1, sizeof( WAVEHDR) );
			sprintf( sztemp, "Dummy : %d", uDummyReads );
			pWnd->SetWindowText( sztemp  );
			uDummyReads--;
			return;
		}

		// Process the audio data read.

		DWORD dwBytesRecorded = ((LPWAVEHDR)dwParam1)->dwBytesRecorded;
		switch( wfx.wBitsPerSample )
		{
			case 24:
				j = 3 * nAdd2ForStereo;
				FilterInit( ptrc[0] + (ptrc[1]<<8) + (ptrc[2]<<16) +( ptrc[2] &0x80? 255<<24: 0 ) );
				for( i = 0; i < dwBytesRecorded; i+= j, ptrc += j)
				{
					x = ptrc[0] + (ptrc[1]<<8) + (ptrc[2]<<16) +( ptrc[2] &0x80? 255<<24: 0 );
					if( pSetup->bUseHighPassFilter )
						x = (__int32)((*pFilter)( x ));
					if( abs( x ) > nMax )
						nMax = x;
					u64sum += (__int64(x) * (x));
					uMean += x;
				}
				dMean = (double)uMean / (dwBytesRecorded/j);
				dMax = ((double)nMax) / (32768*256);
				if( dMean < 0 ) 
					dMean = -dMean;
				dResult = (double)u64sum / (dwBytesRecorded/j);
				dResult = (sqrt( dResult ) - dMean)/(32768*256);
				break;
			case 16:
				j = 2 * nAdd2ForStereo;
				FilterInit( *ptr );
				for( i = 0; i < dwBytesRecorded; i+= j, ptr += nAdd2ForStereo)
				{
					y = *ptr;
					if( pSetup->bUseHighPassFilter )
						y = (__int16)((*pFilter)( y ));
					if( abs( y ) > nMax )
						nMax = y;
					u64sum += (__int32(y) * (y));
					uMean += y;
				}
				dMean = (double)uMean / (dwBytesRecorded/j);
				if( dMean < 0 ) 
					dMean = -dMean;
				dMax = ((double)nMax) / (32768);
				dResult = (double)u64sum / (dwBytesRecorded/j);
				dResult = (sqrt( dResult ) - dMean)/32768;
				break;
			case 8:
				j = nAdd2ForStereo;
				FilterInit( *ptrc );
				for( i = 0; i < dwBytesRecorded; i += j, ptrc += nAdd2ForStereo)
				{
					y = *ptrc;
					if( pSetup->bUseHighPassFilter )
						y = (__int16)((*pFilter)( y ));
					if( y > nMax )
						nMax = y;
					u64sum += (__int16(y) * (y));
					uMean += y;
				}
				dMean = (double)uMean / (dwBytesRecorded/j);
				if( dMean < 0 ) dMean = -dMean;
				dMax = ((double)nMax) / (128);
				dResult = (double)u64sum / (dwBytesRecorded/j);
				dResult = (sqrt( dResult ) - dMean)/128;
				break;
			default:
				ASSERT( FALSE );
		}
// Result may be negative but we do not care as we square it for power
//		if( dResult < 0 ) 
//			dResult = 0-dResult;

		dPowerNF[nDataStoreIndex] = dResult * dResult;

		if( dMax > 0.98 )
		{
			i_am_done = true;
			nResponseCode = IDERRORTOOHIGH;
			dlgWaitMessage.PostMessage(WM_CLOSE);
//			dlgWaitMessage.EndDialog(IDERRORTOOHIGH);
			return;
		}
		nDataStoreIndex++;
		int nAverageCount = pMeasure->m_nAverageCount;

		// Got to the end? If so, close the dialog box with errr code 0
		// If mot, pass buffer back to audio system and wait for next one
		if( !pMeasure->m_bDoAverage || nDataStoreIndex == nAverageCount || i_am_done)
		{
			i_am_done = true;
			nResponseCode = 1;
			dlgWaitMessage.PostMessage(WM_CLOSE);
//			dlgWaitMessage.EndDialog(1);
		}
		else
		{
			MMRESULT mmResult = waveInAddBuffer( hwi, (LPWAVEHDR)dwParam1, sizeof( WAVEHDR) );
			sprintf( sztemp, "Reading %s : %d", sReadingName, nAverageCount-nDataStoreIndex );
			pWnd->SetWindowText( sztemp  );
		}
	}
}

// ********************************************************************************
// When we lose focus, make sure input process is stopped.

BOOL deviceNF::OnKillActive() 
{
	StopInput();

	return CPropertyPage::OnKillActive();
}

// ********************************************************************************
// The following 4 functions are triggered by the user selecting a measurement.
// Each does one or more readings depending on the automation settins.
// The values 0,1,2,3 are HARD CODED - see header to DoReading() and we use
// the bit pattern so I have not put in #define vales to discourage changes.

// In each case the process will loop for as long as cancel is not pressed if the 
// repeat flag is set.

void deviceNF::OnButtonRefOn() 
{
	bool bContinue = TRUE;

	while (bContinue )
	{
		bContinue = DoReading(0, IDC_EDIT_REF_ON);
		if( m_bNoiseSwitchAuto && bContinue)
			bContinue = DoReading(1, IDC_EDIT_REF_OFF);
		if( !bMeasureOverallNoiseFigure && m_bDutSwitchAuto && bContinue)
		{
			bContinue = DoReading(2, IDC_EDIT_DEVICE_ON);
			if( m_bNoiseSwitchAuto && bContinue)
				bContinue = DoReading(3, IDC_EDIT_DEVICE_OFF);
		}
		if( !bMeasureRepeat )
			bContinue = FALSE;
	}
}

void deviceNF::OnButtonRefOff() 
{
	bool bContinue = TRUE;

	while (bContinue )
	{
		bContinue = DoReading(1, IDC_EDIT_REF_OFF);
		if( !bMeasureOverallNoiseFigure && m_bDutSwitchAuto && bContinue)
			bContinue = DoReading(3, IDC_EDIT_DEVICE_OFF);
		if( !bMeasureRepeat )
			bContinue = FALSE;
	}
}


void deviceNF::OnButtonDeviceOn() 
{
	bool bContinue = TRUE;

	while (bContinue )
	{
		bContinue = DoReading(2, IDC_EDIT_DEVICE_ON);
		if( m_bNoiseSwitchAuto && bContinue)
			bContinue = DoReading(3, IDC_EDIT_DEVICE_OFF);
		if( !bMeasureRepeat )
			bContinue = FALSE;
	}
}


void deviceNF::OnButtonDeviceOff() 
{
	bool bContinue = TRUE;

	while( bContinue )
	{
		bContinue = DoReading(3, IDC_EDIT_DEVICE_OFF);
		if( !bMeasureRepeat )
			bContinue = FALSE;
	}
}

// ********************************************************************************
// function called for one of the 4 measurements. These are, in sequence,
//	0	Noise source on, DUT out of circuit
//	1	Noise source off, DUT out of circuit
//	2	Noise source on, DUT in circuit
//	3	Noise source off, DUT in circuit
//
// ResultDialog tells it which window to show the reading in.
// First, make sure input is stopped. 
// Then switch the noise source and device appropriately
// Then set up variables (number of dummy reads counter etc)
// and if the current audio device and settings are valid, start
// the reading process. In the event of errors, show an error box.
// Put up a dialog box while reading (the WaveInProc() will update it)
// We then wait until the WaveInProc() clears the dialog or until the
// user closes it (cancel). The return code tells us which caused it.
// Then stop the input process.
// If needed, average the results, display the reading and show
// the compted results.

// returns flag TRUE unless an error or cancel occurred to allow repeat readings

bool deviceNF::DoReading( int WhichOne, int ResultDialog )
{
	int i;
	CException ex;

	StopInput();

	set_wfx();

	// set port controls
	if( pSetup->bEnableAutomation && (m_bDutSwitchAuto || m_bNoiseSwitchAuto ))
		pSetup->SwitchDevices( (WhichOne & 1) == 0, (WhichOne & 2) != 0 );
	uDummyReads = m_uDummyReads;
	sReadingName.LoadString( ReadingNames[WhichOne] );


	if( IsAudioDeviceValid() )
	{
		i_am_done = false;

		if( StartReading((DWORD)waveGotBlockNF)!= MMSYSERR_NOERROR )
		{
				CException ex;
				ex.ReportError( MB_OK, IDS_CANNOT_OPEN_WAVEIN );
				return FALSE;
		}
	}
	else
		return FALSE;

	nResponseCode = 0;
	// Wait until readings are finished. 
	// This will either be because user clicked CANCEL or because
	// waveInProc killed it with error return 0.
	int nResponse = dlgWaitMessage.DoModal();

	StopInput();

//	if (nResponse == IDERRORTOOHIGH)
	if (nResponseCode == IDERRORTOOHIGH)
	{
			CException ex;
			ex.ReportError( MB_OK, IDS_SIGNAL_TOO_BIG );
			// we return FALSE at the end
	}
//	else if (nResponse != IDCANCEL)
	else if (nResponseCode == 1 )
	{
		// process reading
		dAverageNF[WhichOne] =0;
		if( pMeasure->m_bDoAverage )
		{
			int nAverageCount = pMeasure->m_nAverageCount;
			for(i=0; i< nAverageCount; i++)
				dAverageNF[WhichOne] += dPowerNF[i];
			dAverageNF[WhichOne] /= nAverageCount;
		}
		else
			dAverageNF[WhichOne] = dPowerNF[0];
		CWnd *pWnd = GetDlgItem(ResultDialog);
		ASSERT_VALID(pWnd);
		ShowReading( pWnd, dAverageNF[WhichOne] < (double)1e-12, 10* log10( dAverageNF[WhichOne]), IDS_STRING_ERROR, NULL , TRUE);
		bGotReadingNF[WhichOne] = TRUE;
		ShowResults();

		if( pSetup->bUseLogFile )
		{
			FILE *fp;

			if( (fp = fopen( pSetup->m_LogFilename, "a" )) != NULL )
			{
				CTime theTime = CTime::GetCurrentTime();
				CString t = theTime.Format("%c");
				fprintf(fp,"NF,%s,%s,%f,%s\n", (LPCTSTR)t,(LPCTSTR)sReadingName, 10*log10(dAverageNF[WhichOne]), (LPCTSTR)sLogResults );
				fclose(fp);
			}
			else
			{
				ex.ReportError( MB_OK, IDS_CANNOT_OPEN_LOGFILE );
				return FALSE;
			}
		}
		return TRUE;
	}
	return FALSE;
}

// ********************************************************************************
// Private function used in NF measurement to update the computed results of
// gain and noise.
// called from any place where results may change (for example changing ENR, TSoff)
// or when we have made a reading

void deviceNF::ShowResults()
{
	// the variables and names are as per the Agilent technical note
	double dT1;		// computed noise temperature of the device under test
	double dGain;	// computed gain of DUT
	double dY12;	// Y factor computed for overall system
	double dT12;	// Noise temperature of overall system
	double dENRabs;	// ENR as a ratio (instead of dB)
	double dY2;		// Y factor computed for measuring system
	double dTSon;	// Noise temperature when noise source on
	double dT2;		// Noise temperature for measuring system
	double dF;		// Computed noise factor
	double dNF;		// computed noise figure in dB
	double dGainDB;	// computed gain in dB

	// if you are measuring overall system and have first 2 readings, 
	// or you are not and you have all 4......
	if( bGotReadingNF[0] && bGotReadingNF[1] && ( bMeasureOverallNoiseFigure || (bGotReadingNF[2] && bGotReadingNF[3] )))
	{
		dENRabs = pow((double)10,(dEnr/10));
		dY2 = dAverageNF[0]/dAverageNF[1];
		dTSon = dENRabs*dT0 + dTSOff;
		dT2 = (dTSon - dY2*dTSOff)/(dY2-1);

		// Skip calculation for DUT if measuring overall system
		if( bMeasureOverallNoiseFigure )
		{
			dT1 = dT2;
		}
		else
		{
			dY12 = dAverageNF[2] / dAverageNF[3];
			dGain = ( dAverageNF[2] - dAverageNF[3] ) / (dAverageNF[0] - dAverageNF[1] );
			dT12 = (dTSon - dY12*dTSOff)/(dY12-1);
			dT1 = dT12 - dT2/dGain;
		}

		// calculate results (noise factor, figure and gain in dB)
		dF = 1 + dT1/dT0;
		dNF = 10* log10(dF);
		dGainDB = 10*log10(dGain);

		// OK, all done, if measuring DUT, display its gain
		CWnd *pWnd = GetDlgItem(IDC_EDIT_GAIN);
		ASSERT_VALID(pWnd);
		if( bMeasureOverallNoiseFigure )
		{
			sLogResults.LoadString(IDS_GAIN_NOT_MEASURED);
			pWnd->SetWindowText("");
		}
		else
		{
			sLogResults = "";
			ShowReading( pWnd, dGainDB < -(GAIN_LIMIT_DB) || dGainDB > GAIN_LIMIT_DB, dGainDB, IDS_STRING_ERROR, &fnMyFont24, TRUE );
		}

		// show noise results
		pWnd = GetDlgItem(IDC_EDIT_NTEMP);
		ASSERT_VALID(pWnd);
		ShowReading( pWnd, dT1 < 0 || dT1 > T_LIMIT_K, dT1, IDS_STRING_ERROR, &fnMyFont24, TRUE );

		pWnd = GetDlgItem(IDC_EDIT_NFACTOR);
		ASSERT_VALID(pWnd);
		ShowReading( pWnd, dT1 < 0 || dT1 > T_LIMIT_K, dF, IDS_STRING_ERROR, &fnMyFont24, TRUE );

		pWnd = GetDlgItem(IDC_EDIT_NFIGURE);
		ASSERT_VALID(pWnd);
		ShowReading( pWnd, dT1 < 0 || dT1 > T_LIMIT_K, dNF, IDS_STRING_ERROR, &fnMyFont24, TRUE );
	}
	else
		sLogResults.LoadString(IDS_NOT_CALCULATED);
}

// ********************************************************************************
// User changes the ENR value. Note that this can either be the ENR value or the
// frequency of interest depending on the configuration. We don't care - just call
// the ComputeENR() function and that will sort it out.
// However, if it was a frequency, then we will display the computed ENR value
// as a diagnostic.
// Finally call ShowResults() to update results (if needed)

void deviceNF::OnChangeEditEnr() 
{
	// TODO: If this is a RICHEDIT control, the control will not
	// send this notification unless you override the CPropertyPage::OnInitDialog()
	// function to send the EM_SETEVENTMASK message to the control
	// with the ENM_CHANGE flag ORed into the lParam mask.
	
	char sztemp[128];

	CWnd *pWnd = GetDlgItem(IDC_EDIT_ENR);
	ASSERT_VALID(pWnd);
	pWnd->GetWindowText(sztemp, sizeof(sztemp));
	dEnrOrFrequency = atof( sztemp );
	dEnr = pSetup->ComputeENR( dEnrOrFrequency );

	pWnd = GetDlgItem(IDC_EDIT_ENR2);
	ASSERT_VALID(pWnd);
	if( pSetup->bUseEnrFile )
		sprintf( sztemp, "%.3f", dEnr);
	else
		sztemp[0] = 0;
	pWnd->SetWindowText( sztemp );

	ShowResults();
}

// ********************************************************************************
// Whenever the user changes we get called. Read and set the TSoff variable then
// call ShowResults() to update the displayed calculations (if any to show)

void deviceNF::OnChangeEditTsoff() 
{
	// TODO: If this is a RICHEDIT control, the control will not
	// send this notification unless you override the CPropertyPage::OnInitDialog()
	// function to send the EM_SETEVENTMASK message to the control
	// with the ENM_CHANGE flag ORed into the lParam mask.
	
	char sztemp[128];

	CWnd *pWnd = GetDlgItem(IDC_EDIT_TSOFF);
	ASSERT_VALID(pWnd);
	pWnd->GetWindowText(sztemp, sizeof(sztemp));
	dTSOff = atof( sztemp );
	ShowResults();
}

// ********************************************************************************
// called when the 'average readings' checkbox is ticked or unticked.
// sets boolean flag appropriately

void deviceNF::OnCheckAverage() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_AVERAGE);
	ASSERT_VALID(pWnd);

	pMeasure->m_bDoAverage = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
}

// ********************************************************************************
// Allows user to specify the number of traces to be averaged.
// Sets variable appropriately

void deviceNF::OnSelendokComboAvgCount() 
{
	CWnd* pWnd = GetDlgItem(IDC_COMBO_AVG_COUNT);
	ASSERT_VALID(pWnd);

	pMeasure->m_nAverageCount = pWnd->SendMessage(CB_GETCURSEL , 0, 0) + 1;

}

// ********************************************************************************
// Function called when 'save settings' button is pressed. Save the average count,
// average flag and number of dummy reads.

void deviceNF::OnButtonMeasureSave() 
{
	CString sSetup, sToken;
	sSetup.LoadString(IDS_STRING_SETUP);
	CException ex;
	bool bError = false;

	sToken.LoadString(IDS_TRACE_AVERAGE_COUNT);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, pMeasure->m_nAverageCount ))	
		bError = true;
	sToken.LoadString(IDS_TRACE_AVERAGE_FLAG);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, pMeasure->m_bDoAverage ))
		bError = true;
	sToken.LoadString(IDS_DUMMY_READS);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, m_uDummyReads ))
		bError = true;
	sToken.LoadString(IDS_AUTOMATE_NF_NOISE);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, m_bNoiseSwitchAuto ))
		bError = true;
	sToken.LoadString(IDS_AUTOMATE_NF_DUT);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, m_bDutSwitchAuto ))
		bError = true;
	if( bError )
		ex.ReportError( MB_OK, IDS_CANNOT_WRITE_REGISTRY );
}

// ********************************************************************************
// When the user ticks or unticks the 'overall noise figure' checkbox this is called
// It sets the corresponding flag

void deviceNF::OnCheckOverallNfMode() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_OVERALL_NF_MODE);
	ASSERT_VALID(pWnd);

	bMeasureOverallNoiseFigure = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
	SetButtonConfig();
	ShowResults();
}

// ********************************************************************************
// Function called when the number of dummy reads selection is changed.
// reads value and sets m_uDummyReads.

void deviceNF::OnSelendokComboDummies() 
{
	CWnd* pWnd = GetDlgItem(IDC_COMBO_DUMMIES);
	ASSERT_VALID(pWnd);
	m_uDummyReads = pWnd->SendMessage(CB_GETCURSEL , 0, 0);
	
}

// ********************************************************************************
// Function called when the repeat checkbox is changed.
// reads value and sets bMeasureRepeat.

void deviceNF::OnCheckRepeat() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_REPEAT);
	ASSERT_VALID(pWnd);

	bMeasureRepeat = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
}

void deviceNF::OnButtonClr() 
{
	bGotReadingNF[0] = bGotReadingNF[1] = bGotReadingNF[2] = bGotReadingNF[3] = FALSE;	

	CWnd* pWnd = GetDlgItem(IDC_EDIT_REF_ON);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_REF_OFF);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_DEVICE_ON);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_DEVICE_OFF);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");

	pWnd = GetDlgItem(IDC_EDIT_NTEMP);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_NFACTOR);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_NFIGURE);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_GAIN);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");

}
