// common.cpp : implementation file
//
//(C) Copyright Dave Roberts G8KBB 2005
// This program is released only for the purpose of self training and eduation
// in amatuer radio. It may not be used for any commercial purpose, sold or
// modified.
// It has been written as an exercise in self tuition by me as part of my hobby
// and I make no claims about its fitness for purpose or correct operation.

#include "stdafx.h"
#include "NoiseMeter.h"
#include "deviceNF.h"
#include "measure.h"

#include "mmsystem.h"

#include "spectrum.h"
#include "setup.h"

#include "common.h"

#include "math.h"

#include "WaitingForMeasurement.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


extern spectrum *pSpectrum;
extern measure *pMeasure;
extern setup *pSetup;

extern CString sLogResults;


// index into data store for current/next reading
int nDataStoreIndex;
// boolean flag set to TRUE when audio input is active
bool bRunning;

// Set of structures needed for handling waveform input
//
// we need a handle to the audio device
HWAVEIN hwi;
// we need to control waveform format
WAVEFORMATEX wfx;
// and we need 2 audio buffer headers, one for each audio buffer
WAVEHDR wh1, wh2;
// two input buffer for current reading passed to drivers
char cInBufAudio1[MAX_DATA_BUFFER_SIZE];
char cInBufAudio2[MAX_DATA_BUFFER_SIZE];

// ********************************************************************************
// This function is common to the audio processes.
// we give it an address of a WaveInProc() and it starts the audio input.
// Two versions of the function - one that just takes an address of a WaveInProc,
// which uses the 'default' block size based on sample rate and time.
// The other passes in a block size (used by FFT routine)
// Returns 0 in the case of no error or an error code otherwise.

int StartReading(DWORD WaveInProc)
{
	ASSERT( pSetup->nMilliSeconds <= MAX_TIME_MSEC );
	return StartReading( WaveInProc, wfx.nAvgBytesPerSec * pSetup->nMilliSeconds / 1000 );
}


int StartReading(DWORD WaveInProc, DWORD dDesiredBlockSize)
{
	if( bRunning)
		StopInput();
	memset( &wh1, 0, sizeof(WAVEHDR)); 
	memset( &wh2, 0, sizeof(WAVEHDR)); 
	wh1.lpData = cInBufAudio1;
	wh2.lpData = cInBufAudio2;
	wh1.dwBufferLength = dDesiredBlockSize ; 
	wh2.dwBufferLength = dDesiredBlockSize ; 
	ASSERT( dDesiredBlockSize  <= MAX_DATA_BUFFER_SIZE );
	nDataStoreIndex = 0;

	MMRESULT mmResult = waveInOpen( &hwi, pSetup->m_uWaveInDev, &wfx, WaveInProc, NULL, CALLBACK_FUNCTION );
	if( mmResult != MMSYSERR_NOERROR )
		return mmResult;
	mmResult = waveInPrepareHeader( hwi, &wh1, sizeof( wh1) );
	if( mmResult != MMSYSERR_NOERROR )
	{
		waveInClose( hwi );
		return mmResult;
	}
	mmResult = waveInPrepareHeader( hwi, &wh2, sizeof( wh2) );
	if( mmResult != MMSYSERR_NOERROR )
	{
		waveInUnprepareHeader( hwi, &wh1, sizeof( wh1) );
		waveInClose( hwi );
		return mmResult;
	}
	mmResult = waveInAddBuffer( hwi, &wh1, sizeof( wh1) );
	if( mmResult == MMSYSERR_NOERROR )
		mmResult = waveInAddBuffer( hwi, &wh2, sizeof( wh2) );
	bRunning = TRUE;
	if( mmResult == MMSYSERR_NOERROR )
		mmResult = waveInStart( hwi );

	if( mmResult != MMSYSERR_NOERROR )
	{
		waveInUnprepareHeader( hwi, &wh1, sizeof( wh1) );
		waveInUnprepareHeader( hwi, &wh2, sizeof( wh2) );
		waveInClose( hwi );
	}
	return mmResult;
}

// ********************************************************************************
// This is the corresponding function to stop audio input and free the buffers
//

BOOL StopInput()
{
	if( bRunning == TRUE)
	{
		bRunning = FALSE;
		// ignore error returns
		MMRESULT mmResult = waveInReset( hwi );
		ASSERT( mmResult == MMSYSERR_NOERROR );
		mmResult = waveInUnprepareHeader( hwi, &wh1, sizeof( wh1) );
		ASSERT( mmResult == MMSYSERR_NOERROR );
		mmResult = waveInUnprepareHeader( hwi, &wh2, sizeof( wh2) );
		ASSERT( mmResult == MMSYSERR_NOERROR );
		mmResult = waveInClose( hwi );
		ASSERT( mmResult == MMSYSERR_NOERROR );
		return TRUE;
	}
	else
		return FALSE;
}

// ********************************************************************************
// A simple function to determine if we can access the audio device as requested
//

BOOL IsAudioDeviceValid()
{
	MMRESULT mmResult = waveInOpen( &hwi, pSetup->m_uWaveInDev, &wfx, NULL, NULL, CALLBACK_NULL );
	if( mmResult == MMSYSERR_NOERROR )
		waveInClose( hwi );
	return ( mmResult == MMSYSERR_NOERROR );
}

// ********************************************************************************
// A helper function to set the waveform format structure
//

void set_wfx()
{
	wfx.wFormatTag = WAVE_FORMAT_PCM; // Formats[m_uWaveInFormat].formatid;
	wfx.nChannels = pSetup->bIsStereo? 2: 1;
	wfx.nSamplesPerSec = pSetup->nSampleRate; //nSampleTable[pSetup->m_uWaveInFormat];
	ASSERT( wfx.nSamplesPerSec <= MAX_ADC_SPEED );
	wfx.wBitsPerSample = pSetup->nBitsPerSample;
	wfx.nBlockAlign = (wfx.nChannels * wfx.wBitsPerSample)/8;
	wfx.nAvgBytesPerSec = wfx.nSamplesPerSec * wfx.nBlockAlign;
	wfx.cbSize = 0;
}

// ********************************************************************************
// This helper is called to display a result in a window.
// It is given a pointer to the window with pWnd.
// bIsError is a flag that determines whether the required value or an error message is shown
// uErrorMessage is a string handle.
// If the pointer to font is not NULL, it must point to the required font.
// If bLogFlag is TRUE, add result to collated results string for logging
//    Parameter is optional, default is no logging

void ShowReading( CWnd *pWnd, BOOL bIsError, double dValue, int uErrorMessage, CFont *font, bool bLogFlag )
{
	CString sMessage;

	ASSERT_VALID( pWnd );

	if( bIsError )
		sMessage.LoadString( uErrorMessage);
	else
		switch( pSetup->nAutomateDecimalPoints )
		{
			case 0:
				sMessage.Format(" %.0f", dValue );
				break;
			case 1:
				sMessage.Format(" %.1f", dValue );
				break;
			case 2:
				sMessage.Format(" %.2f", dValue );
				break;
			default:
				sMessage.Format(" %.3f", dValue );
				break;
		}
	if( font != NULL )
		pWnd->SetFont( font, 0 );

	pWnd->SetWindowText( sMessage );

	if( bLogFlag && pSetup->bUseLogFile )
		sLogResults += sMessage+",";
}

