// CalibrateNoiseSource.cpp : implementation file
//
//(C) Copyright Dave Roberts G8KBB 2005
// This program is released only for the purpose of self training and eduation
// in amatuer radio. It may not be used for any commercial purpose, sold or
// modified.
// It has been written as an exercise in self tuition by me as part of my hobby
// and I make no claims about its fitness for purpose or correct operation.

#include "stdafx.h"
#include "NoiseMeter.h"
#include "CalibrateNoiseSource.h"
#include "deviceNF.h"
#include "measure.h"

#include "mmsystem.h"

#include "spectrum.h"
#include "setup.h"

#include "common.h"

#include "math.h"

#include "WaitingForMeasurement.h"

#include "high-pass-filters.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// Pointers to other objects
extern spectrum *pSpectrum;
extern measure *pMeasure;
extern setup *pSetup;
extern deviceNF *pDeviceNF;

// storage for 4 measurements made for ENR measurement
// In sequence these are calibrated source on and off then
// uncalibrated source on and off
double dAverageEnr[4];
// boolean flags showing which of the above readings are valid
bool bGotReadingEnr[4];
// How many dummy reads before valid reads to allow settling time?
extern int uDummyReads;
// string used to put reading name into dialog 
CString sReadingNameEnr;
// and index for strings used
int ReadingNamesEnr[] =
{
	IDS_CAL_NOISE_ON,
	IDS_CAL_NOISE_OFF,
	IDS_TARGET_NOISE_ON,
	IDS_TARGET_NOISE_OFF
};
// Dialog shown whilst reading - we want to point to it to update text on it
extern WaitingForMeasurement dlgWaitMessage;
extern int nResponseCode;
// storage for computed power readings
double dPowerENR[MAX_AVERAGE];
// fonts used other than system font in the displays
// no longer ise font size 20
//extern CFont fnMyFont20;
extern CFont fnMyFont24;
// string used in logging to create calculated results
extern CString sLogResults;

void CALLBACK waveGotBlockEnr(
  HWAVEIN hwi,       
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2 );


/////////////////////////////////////////////////////////////////////////////
// CCalibrateNoiseSource property page

IMPLEMENT_DYNCREATE(CCalibrateNoiseSource, CPropertyPage)

CCalibrateNoiseSource::CCalibrateNoiseSource() : CPropertyPage(CCalibrateNoiseSource::IDD)
{
	//{{AFX_DATA_INIT(CCalibrateNoiseSource)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT

	CString sSetup, sToken;
	sSetup.LoadString(IDS_STRING_SETUP);

	sToken.LoadString(IDS_TSOFF);
	CString szTemp = AfxGetApp( )->GetProfileString( sSetup, sToken, DEFAULT_TSOFF );
	dTSOffCalibrated = dTSOffTarget = atof( szTemp );

	sToken.LoadString(IDS_AUTOMATE_CAL_NOISE);
	m_bNoiseSwitchAuto = AfxGetApp( )->GetProfileInt( sSetup, sToken, FALSE ) & 1;

	bMeasureRepeat = FALSE;
}

CCalibrateNoiseSource::~CCalibrateNoiseSource()
{
}

void CCalibrateNoiseSource::DoDataExchange(CDataExchange* pDX)
{
	CPropertyPage::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CCalibrateNoiseSource)
		// NOTE: the ClassWizard will add DDX and DDV calls here
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CCalibrateNoiseSource, CPropertyPage)
	//{{AFX_MSG_MAP(CCalibrateNoiseSource)
	ON_BN_CLICKED(IDC_CHECK_AVERAGE, OnCheckAverage)
	ON_BN_CLICKED(IDC_CHECK_NOISE_SWITCH, OnCheckNoiseSwitch)
	ON_BN_CLICKED(IDC_CHECK_REPEAT, OnCheckRepeat)
	ON_CBN_SELENDOK(IDC_COMBO_DUMMIES, OnSelendokComboDummies)
	ON_CBN_SELENDOK(IDC_COMBO_AVG_COUNT, OnSelendokComboAvgCount)
	ON_BN_CLICKED(IDC_BUTTON_MEASURE_SAVE, OnButtonMeasureSave)
	ON_EN_CHANGE(IDC_EDIT_TSOFF_CALIBRATED, OnChangeEditTsoffCalibrated)
	ON_EN_CHANGE(IDC_EDIT_TSOFF_TARGET, OnChangeEditTsoffTarget)
	ON_BN_CLICKED(IDC_BUTTON_REF_ON, OnButtonRefOn)
	ON_BN_CLICKED(IDC_BUTTON_REF_OFF, OnButtonRefOff)
	ON_BN_CLICKED(IDC_BUTTON_DEVICE_ON, OnButtonDeviceOn)
	ON_BN_CLICKED(IDC_BUTTON_DEVICE_OFF, OnButtonDeviceOff)
	ON_EN_CHANGE(IDC_EDIT_ENR, OnChangeEditEnr)
	ON_BN_CLICKED(IDC_BUTTON_CLR, OnButtonClr)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CCalibrateNoiseSource message handlers

BOOL CCalibrateNoiseSource::OnInitDialog() 
{
	CPropertyPage::OnInitDialog();
	
	int nCount;
	char szTemp[128];

	CWnd* pWnd = GetDlgItem(IDC_COMBO_AVG_COUNT);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	for( nCount =1; nCount < MAX_AVERAGE; nCount++)
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)_itoa(nCount,szTemp,10)));

	pWnd = GetDlgItem(IDC_COMBO_DUMMIES);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_RESETCONTENT, 0, 0);
	for( nCount =0; nCount <= MAX_DUMMY_READS; nCount++)
		pWnd->SendMessage(CB_ADDSTRING, 0, (LPARAM)((LPCTSTR)_itoa(nCount,szTemp,10)));
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)pDeviceNF->m_uDummyReads, 0);

	bGotReadingEnr[0] = bGotReadingEnr[1] = bGotReadingEnr[2] = bGotReadingEnr[3] = FALSE;	

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}

BOOL CCalibrateNoiseSource::OnSetActive() 
{
	CString sTemp;
	char sztemp[128];
	CWnd *pWnd;

	SetButtonConfig();

	pWnd = GetDlgItem(IDC_EDIT_ENR);
	ASSERT_VALID(pWnd);
	sprintf( sztemp, "%.3f", pDeviceNF->dEnrOrFrequency);
	pWnd->SetWindowText( sztemp );
	pWnd = GetDlgItem(IDC_EDIT_TSOFF_CALIBRATED);
	ASSERT_VALID(pWnd);
	sprintf( sztemp, "%.2f", dTSOffCalibrated);
	pWnd->SetWindowText( sztemp );
	pWnd = GetDlgItem(IDC_EDIT_TSOFF_TARGET);
	ASSERT_VALID(pWnd);
	sprintf( sztemp, "%.2f", dTSOffTarget);
	pWnd->SetWindowText( sztemp );
	pWnd = GetDlgItem(IDC_CHECK_AVERAGE);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(BM_SETCHECK, pMeasure->m_bDoAverage, 0);

	pWnd = GetDlgItem(IDC_EDIT_T0);
	ASSERT_VALID(pWnd);
	sprintf( sztemp, "%.2f", pDeviceNF->dT0);
	pWnd->SetWindowText( sztemp );

	SET_STATE( IDC_STATIC_ENR_FROM_F, pSetup->bUseEnrFile );

	pWnd = GetDlgItem(IDC_STATIC_USE_F);
	ASSERT_VALID(pWnd);
	sTemp.LoadString( pSetup->bUseEnrFile ? IDS_STRINGNF_ENR_F : IDS_STRINGNF_ENR_DB );
	pWnd->SetWindowText(sTemp);

	pWnd = GetDlgItem(IDC_COMBO_AVG_COUNT);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(CB_SETCURSEL, (WPARAM)pMeasure->m_nAverageCount-1, 0);

	pWnd = GetDlgItem(IDC_CHECK_NOISE_SWITCH);
	ASSERT_VALID(pWnd);
	pWnd->SendMessage(BM_SETCHECK, m_bNoiseSwitchAuto, 0);

	return CPropertyPage::OnSetActive();
}

// ********************************************************************************
// Helper utility to set the configuration of on screen buttons etc.
// enable or disable window controls as needed

void CCalibrateNoiseSource::SetButtonConfig()
{
	CWnd *pWnd;

	pWnd = GetDlgItem(IDC_CHECK_NOISE_SWITCH);
	ASSERT_VALID(pWnd);
	pWnd->EnableWindow( pSetup->bEnableAutomation);
	if( pSetup->bEnableAutomation == FALSE)
		m_bNoiseSwitchAuto = FALSE;

	SET_STATE( IDC_BUTTON_REF_OFF, !m_bNoiseSwitchAuto );

	SET_STATE( IDC_BUTTON_DEVICE_OFF, !m_bNoiseSwitchAuto );

}

// ********************************************************************************
// when user wants to average readings, set flag (shared between functions)

void CCalibrateNoiseSource::OnCheckAverage() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_AVERAGE);
	ASSERT_VALID(pWnd);

	pMeasure->m_bDoAverage = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
}

// ********************************************************************************
// handle selection of automation. When changed, update button status accordingly

void CCalibrateNoiseSource::OnCheckNoiseSwitch() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_NOISE_SWITCH);
	ASSERT_VALID(pWnd);

	m_bNoiseSwitchAuto = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
	SetButtonConfig();
	
}

// ********************************************************************************
// toggle function to continually repeat readings

void CCalibrateNoiseSource::OnCheckRepeat() 
{
	CWnd* pWnd = GetDlgItem(IDC_CHECK_REPEAT);
	ASSERT_VALID(pWnd);

	bMeasureRepeat = pWnd->SendMessage(BM_GETCHECK, 0, 0) == BST_CHECKED ? TRUE : FALSE;
}

// ********************************************************************************
// when user changes selection for number of dummy reads

void CCalibrateNoiseSource::OnSelendokComboDummies() 
{
	CWnd* pWnd = GetDlgItem(IDC_COMBO_DUMMIES);
	ASSERT_VALID(pWnd);
	pDeviceNF->m_uDummyReads = pWnd->SendMessage(CB_GETCURSEL , 0, 0);
}

// ********************************************************************************
// user changes number of traces to average (does not affect average enabled flag)

void CCalibrateNoiseSource::OnSelendokComboAvgCount() 
{
	CWnd* pWnd = GetDlgItem(IDC_COMBO_AVG_COUNT);
	ASSERT_VALID(pWnd);

	pMeasure->m_nAverageCount = pWnd->SendMessage(CB_GETCURSEL , 0, 0) + 1;
}

// ********************************************************************************
// save config in registry. We only change average and dummy info

void CCalibrateNoiseSource::OnButtonMeasureSave() 
{
	CException ex;
	bool bError = false;
	CString sSetup, sToken;
	sSetup.LoadString(IDS_STRING_SETUP);

	sToken.LoadString(IDS_TRACE_AVERAGE_COUNT);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, pMeasure->m_nAverageCount ))	
		bError = true;
	sToken.LoadString(IDS_TRACE_AVERAGE_FLAG);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, pMeasure->m_bDoAverage ))
		bError = true;
	sToken.LoadString(IDS_DUMMY_READS);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, pDeviceNF->m_uDummyReads ))
		bError = true;
	sToken.LoadString(IDS_AUTOMATE_CAL_NOISE);
	if( !AfxGetApp( )->WriteProfileInt( sSetup, sToken, m_bNoiseSwitchAuto ))
		bError = true;
	if( bError )
		ex.ReportError( MB_OK, IDS_CANNOT_WRITE_REGISTRY );
}

// ********************************************************************************
// a change to the calibrated device TSoff setting. Read and compute

void CCalibrateNoiseSource::OnChangeEditTsoffCalibrated() 
{
	// TODO: If this is a RICHEDIT control, the control will not
	// send this notification unless you override the CPropertyPage::OnInitDialog()
	// function to send the EM_SETEVENTMASK message to the control
	// with the ENM_CHANGE flag ORed into the lParam mask.
	
	char sztemp[128];

	CWnd *pWnd = GetDlgItem(IDC_EDIT_TSOFF_CALIBRATED);
	ASSERT_VALID(pWnd);
	pWnd->GetWindowText(sztemp, sizeof(sztemp));
	dTSOffCalibrated = atof( sztemp );
	ShowResults();
}

// ********************************************************************************
// a change to the target device TSoff setting. Read and compute

void CCalibrateNoiseSource::OnChangeEditTsoffTarget() 
{
	// TODO: If this is a RICHEDIT control, the control will not
	// send this notification unless you override the CPropertyPage::OnInitDialog()
	// function to send the EM_SETEVENTMASK message to the control
	// with the ENM_CHANGE flag ORed into the lParam mask.
	
	// with the ENM_CHANGE flag ORed into the lParam mask.
	
	char sztemp[128];

	CWnd *pWnd = GetDlgItem(IDC_EDIT_TSOFF_TARGET);
	ASSERT_VALID(pWnd);
	pWnd->GetWindowText(sztemp, sizeof(sztemp));
	dTSOffTarget = atof( sztemp );
	ShowResults();
}

// ********************************************************************************
// The following 4 functions are triggered by the user selecting a measurement.
// Each does one or more readings depending on the automation settins.
// The values 0,1,2,3 are HARD CODED - see header to DoReading() and we use
// the bit pattern so I have not put in #define vales to discourage changes.

// In each case the process will loop for as long as cancel is not pressed if the 
// repeat flag is set.

void CCalibrateNoiseSource::OnButtonRefOn() 
{
	bool bContinue = TRUE;

	while (bContinue )
	{
		bContinue = DoReading(0, IDC_EDIT_REF_ON);
		if( m_bNoiseSwitchAuto && bContinue)
			bContinue = DoReading(1, IDC_EDIT_REF_OFF);
		if( !bMeasureRepeat )
			bContinue = FALSE;
	}
}

void CCalibrateNoiseSource::OnButtonRefOff() 
{
	bool bContinue = TRUE;

	while (bContinue )
	{
		bContinue = DoReading(1, IDC_EDIT_REF_OFF);
		if( !bMeasureRepeat )
			bContinue = FALSE;
	}
}

void CCalibrateNoiseSource::OnButtonDeviceOn() 
{
	bool bContinue = TRUE;

	while (bContinue )
	{
		bContinue = DoReading(2, IDC_EDIT_DEVICE_ON);
		if( m_bNoiseSwitchAuto && bContinue)
			bContinue = DoReading(3, IDC_EDIT_DEVICE_OFF);
		if( !bMeasureRepeat )
			bContinue = FALSE;
	}
}

void CCalibrateNoiseSource::OnButtonDeviceOff() 
{
	bool bContinue = TRUE;

	while( bContinue )
	{
		bContinue = DoReading(3, IDC_EDIT_DEVICE_OFF);
		if( !bMeasureRepeat )
			bContinue = FALSE;
	}
}

// ********************************************************************************
// function called for one of the 4 measurements. These are, in sequence,
//	0	Calibrated Noise source on
//	1	Calibrated Noise source off
//	2	Unknown Noise source on
//	3	Unknown Noise source off
//
// ResultDialog tells it which window to show the reading in.
// First, make sure input is stopped. 
// Then switch the noise source and device appropriately
// Then set up variables (number of dummy reads counter etc)
// and if the current audio device and settings are valid, start
// the reading process. In the event of errors, show an error box.
// Put up a dialog box while reading (the WaveInProc() will update it)
// We then wait until the WaveInProc() clears the dialog or until the
// user closes it (cancel). The return code tells us which caused it.
// Then stop the input process.
// If needed, average the results, display the reading and show
// the compted results.

// returns flag TRUE unless an error or cancel occurred to allow repeat readings

bool CCalibrateNoiseSource::DoReading( int WhichOne, int ResultDialog )
{
	int i;
	CException ex;

	StopInput();

	set_wfx();

	// set port controls
	if( pSetup->bEnableAutomation && (m_bNoiseSwitchAuto ))
		pSetup->SwitchDevices( (WhichOne & 1) == 0, (WhichOne &2) != 0 );
	uDummyReads = pDeviceNF->m_uDummyReads;
	sReadingNameEnr.LoadString( ReadingNamesEnr[WhichOne] );

	if( IsAudioDeviceValid() )
	{
		if( StartReading((DWORD)waveGotBlockEnr)!= MMSYSERR_NOERROR )
		{
				CException ex;
				ex.ReportError( MB_OK, IDS_CANNOT_OPEN_WAVEIN );
				return FALSE;
		}
	}
	else
		return FALSE;

	nResponseCode = 0;
	// Wait until readings are finished. 
	// This will either be because user clicked CANCEL or because
	// waveInProc killed it with error return 0.
	
	int nResponse = dlgWaitMessage.DoModal();

	StopInput();

	if (nResponseCode == IDERRORTOOHIGH)
//	if (nResponse == IDERRORTOOHIGH)
	{
			CException ex;
			ex.ReportError( MB_OK, IDS_SIGNAL_TOO_BIG );
			// we return FALSE at the end
	}
	else if (nResponseCode == 1)
//	else if (nResponse != IDCANCEL)
	{
		// process reading
		dAverageEnr[WhichOne] =0;
		if( pMeasure->m_bDoAverage )
		{
			int nAverageCount = pMeasure->m_nAverageCount;
			for(i=0; i< nAverageCount; i++)
				dAverageEnr[WhichOne] += dPowerENR[i];
			dAverageEnr[WhichOne] /= nAverageCount;
		}
		else
			dAverageEnr[WhichOne] = dPowerENR[0];
		CWnd *pWnd = GetDlgItem(ResultDialog);
		ASSERT_VALID(pWnd);
		ShowReading( pWnd, dAverageEnr[WhichOne] < (double)1e-12, 10* log10( dAverageEnr[WhichOne]), IDS_STRING_ERROR, NULL , TRUE);
		bGotReadingEnr[WhichOne] = TRUE;
		ShowResults();

		if( pSetup->bUseLogFile )
		{
			FILE *fp;

			if( (fp = fopen( pSetup->m_LogFilename, "a" )) != NULL )
			{
				CTime theTime = CTime::GetCurrentTime();
				CString t = theTime.Format("%c");
				fprintf(fp,"ENR,%s,%s,%f,%s\n", (LPCTSTR)t,(LPCTSTR)sReadingNameEnr, 10*log10(dAverageEnr[WhichOne]), (LPCTSTR)sLogResults );
				fclose(fp);
			}
			else
			{
				ex.ReportError( MB_OK, IDS_CANNOT_OPEN_LOGFILE );
				return FALSE;
			}
		}
		return TRUE;
	}
	return FALSE;
}

// ********************************************************************************
// Private function used in ENR measurement to update the computed results.
// called from any place where results may change (for example changing ENR, TSoff)
// or when we have made a reading

void CCalibrateNoiseSource::ShowResults()
{
	double dYcal;	// Y factor for calibrated source
	double dYdev;	// Y factor for unknown noise source
	double dENRcalAbs;	// ENR as a ratio (instead of dB)
	double dENRdev;	// calculated ENR of device

	if( bGotReadingEnr[0] && bGotReadingEnr[1] && bGotReadingEnr[2] && bGotReadingEnr[3] )
	{
		dENRcalAbs = pow((double)10,(dEnrCal/10));
		dYcal = dAverageEnr[0]/dAverageEnr[1];
		dYdev = dAverageEnr[2]/dAverageEnr[3];

		dENRdev = 10*log10((dYdev-1)*(dENRcalAbs/(dYcal-1)-(dTSOffTarget-dTSOffCalibrated)/pDeviceNF->dT0));

		// OK, all done, if measuring DUT, display its ENR
		CWnd *pWnd = GetDlgItem(IDC_EDIT_COMPUTED_ENR);
		ASSERT_VALID(pWnd);
		sLogResults = "";
		ShowReading( pWnd, dENRdev < 0 || dENRdev > ENR_LIMIT_DB, dENRdev, IDS_STRING_ERROR, &fnMyFont24, TRUE );
	}
	else
		sLogResults.LoadString(IDS_NOT_CALCULATED);
}

// ********************************************************************************
// WaveInProc() for the noise figure measurement.
// It is called by the audio subsystem. 

void CALLBACK waveGotBlockEnr(
  HWAVEIN hwi,       
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2 )    
{
	// we only handle the WIM_DATA messages and only if audio input is enabled

	if( uMsg == WIM_DATA && bRunning == TRUE)
	{
		char sztemp[128];
		__int64 u64sum = 0;
		double dResult;
		SHORT *ptr = (short *)(((LPWAVEHDR)dwParam1)->lpData);
		unsigned char *ptrc = (unsigned char *)(((LPWAVEHDR)dwParam1)->lpData);
		DWORD i, j;
		int nMax = 0;
		double dMax;
		int nAdd2ForStereo = wfx.nChannels == 1? 1 : 2;
		double dMean;
		int uMean = 0;
		__int32 x;
		__int16 y;

		CWnd *pWnd = dlgWaitMessage.GetDlgItem(IDC_STATIC_PAUSE);
		ASSERT_VALID(pWnd);

		// handle dummy reads. If any still to do, then simple add the buffer
		// back in again, update the onscreen display and decrement the dummy count
		if( uDummyReads > 0 )
		{
			MMRESULT mmResult = waveInAddBuffer( hwi, (LPWAVEHDR)dwParam1, sizeof( WAVEHDR) );
			sprintf( sztemp, "Dummy : %d", uDummyReads );
			pWnd->SetWindowText( sztemp  );
			uDummyReads--;
			return;
		}

		// Process the audio data read.

		DWORD dwBytesRecorded = ((LPWAVEHDR)dwParam1)->dwBytesRecorded;
		switch( wfx.wBitsPerSample )
		{
			case 24:
				j = 3 * nAdd2ForStereo;
				FilterInit( ptrc[0] + (ptrc[1]<<8) + (ptrc[2]<<16) +( ptrc[2] &0x80? 255<<24: 0 ) );
				for( i = 0; i < dwBytesRecorded; i+= j, ptrc += j)
				{
					x = ptrc[0] + (ptrc[1]<<8) + (ptrc[2]<<16) +( ptrc[2] &0x80? 255<<24: 0 );
					if( pSetup->bUseHighPassFilter )
						x = (__int32)((*pFilter)( x ));
					if( abs( x ) > nMax )
						nMax = x;
					u64sum += (__int64(x) * (x));
					uMean += x;
				}
				dMean = (double)uMean / (dwBytesRecorded/j);
				dMax = ((double)nMax) / (32768*256);
				if( dMean < 0 ) 
					dMean = -dMean;
				dResult = (double)u64sum / (dwBytesRecorded/j);
				dResult = (sqrt( dResult ) - dMean)/(32768*256);
				break;
			case 16:
				j = 2 * nAdd2ForStereo;
				FilterInit( *ptr );
				for( i = 0; i < dwBytesRecorded; i+= j, ptr += nAdd2ForStereo)
				{
					y = *ptr;
					if( pSetup->bUseHighPassFilter )
						y = (__int16)((*pFilter)( y ));
					if( abs( y ) > nMax )
						nMax = y;
					u64sum += (__int32(y) * (y));
					uMean += y;
				}
				dMean = (double)uMean / (dwBytesRecorded/j);
				if( dMean < 0 ) 
					dMean = -dMean;
				dMax = ((double)nMax) / (32768);
				dResult = (double)u64sum / (dwBytesRecorded/j);
				dResult = (sqrt( dResult ) - dMean)/32768;
				break;
			case 8:
				j = nAdd2ForStereo;
				FilterInit( *ptrc );
				for( i = 0; i < dwBytesRecorded; i += j, ptrc += nAdd2ForStereo)
				{
					y = *ptrc;
					if( pSetup->bUseHighPassFilter )
						y = (__int16)((*pFilter)( y ));
					if( y > nMax )
						nMax = y;
					u64sum += (__int16(y) * (y));
					uMean += y;
				}
				dMean = (double)uMean / (dwBytesRecorded/j);
				if( dMean < 0 ) dMean = -dMean;
				dMax = ((double)nMax) / (128);
				dResult = (double)u64sum / (dwBytesRecorded/j);
				dResult = (sqrt( dResult ) - dMean)/128;
				break;
			default:
				ASSERT( FALSE );
		}
// Result may be negative but we do not care as we square it for power
//		if( dResult < 0 ) 
//			dResult = 0-dResult;

		dPowerENR[nDataStoreIndex] = dResult * dResult;

		if( dMax > 0.98 )
		{
			nResponseCode = IDERRORTOOHIGH;
			dlgWaitMessage.PostMessage(WM_CLOSE);
//			dlgWaitMessage.EndDialog(IDERRORTOOHIGH);
			return;
		}
		nDataStoreIndex++;
		int nAverageCount = pMeasure->m_nAverageCount;

		// Got to the end? If so, close the dialog box with errr code 0
		// If mot, pass buffer back to audio system and wait for next one

		if( !pMeasure->m_bDoAverage || nDataStoreIndex == nAverageCount )
		{
			nResponseCode = 1;
			dlgWaitMessage.PostMessage(WM_CLOSE);
//			dlgWaitMessage.EndDialog(0);
		}
		else
		{
			MMRESULT mmResult = waveInAddBuffer( hwi, (LPWAVEHDR)dwParam1, sizeof( WAVEHDR) );
			sprintf( sztemp, "Reading %s : %d", sReadingNameEnr, nAverageCount-nDataStoreIndex );
			pWnd->SetWindowText( sztemp  );
		}
	}
}


void CCalibrateNoiseSource::OnChangeEditEnr() 
{
	// TODO: If this is a RICHEDIT control, the control will not
	// send this notification unless you override the CPropertyPage::OnInitDialog()
	// function to send the EM_SETEVENTMASK message to the control
	// with the ENM_CHANGE flag ORed into the lParam mask.
	
	char sztemp[128];

	CWnd *pWnd = GetDlgItem(IDC_EDIT_ENR);
	ASSERT_VALID(pWnd);
	pWnd->GetWindowText(sztemp, sizeof(sztemp));
	pDeviceNF->dEnrOrFrequency = atof( sztemp );
	dEnrCal = pSetup->ComputeENR( pDeviceNF->dEnrOrFrequency );

	pWnd = GetDlgItem(IDC_EDIT_ENR2);
	ASSERT_VALID(pWnd);
	if( pSetup->bUseEnrFile )
		sprintf( sztemp, "%.3f", dEnrCal);
	else
		sztemp[0] = 0;
	pWnd->SetWindowText( sztemp );

	ShowResults();
}

void CCalibrateNoiseSource::OnButtonClr() 
{
	bGotReadingEnr[0] = bGotReadingEnr[1] = bGotReadingEnr[2] = bGotReadingEnr[3] = FALSE;	

	CWnd* pWnd = GetDlgItem(IDC_EDIT_REF_ON);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_REF_OFF);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_DEVICE_ON);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_DEVICE_OFF);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
	pWnd = GetDlgItem(IDC_EDIT_COMPUTED_ENR);
	ASSERT_VALID(pWnd);
	pWnd->SetWindowText("");
}
