// CustomVNAinterfaceDLL.cpp : Defines the initialization routines for the DLL.
//
// Copyright Dave Roberts G8KBB 2008.
// This program has been written by me as part of my hobby as an educational exercise.
// If you are a Radio Amateur or studentyou are free to use this with a homebrew VNA
// such as the N2PK.
// You may also modify it to make improvements but the above limitations apply to any
// derived products.
// You may not sell it or products derived from it without agreement in advance with me.
//
// As it is an amateur product, no warranty is implied or given.

// The DLL built from this project should be placed in the installation directory of myVNA.
// It may be used to build a custom hardware VNA interface for myVNA to drive.
// maintain the 'def file export definitions and the current project build settings for compatibility


#include "stdafx.h"
#include <afxdllx.h>

// this header file is hardware specific - replace it by your own as needed
#include "vnaio.h"
// this header AND THE DEF FILE defines the interface to myVNA
#include "CustomVNAinterfaceDLL.h"

// This class encapsulates the low level  interface. 
// currently very simple, it contains one object, a pointer
// to a VNA device which is created and destroyed
// modify as you see fit

class CustomVNAhw
{
public:
	VNADevice *dVNA;
	CustomVNAhw()
	{
		dVNA = new VNADevice;
	}
	~CustomVNAhw()
	{
		delete dVNA;
	}
};


// defintions used manipulating phase for the N2PK VNA hardware
// comment out this definition to pass as multiples of phase step
// of remove commenting to pass as multiple of phase step values

#define PASS_DEGREES

// the AD9851 supports 5 bits of phase
#define PHASE_BITS 5
#define PHASEMASK ((1<<PHASE_BITS)-1)

#ifdef _MANAGED
#error Please read instructions in CustomVNAinterface.cpp to compile with /clr
// If you want to add /clr to your project you must do the following:
//	1. Remove the above include for afxdllx.h
//	2. Add a .cpp file to your project that does not have /clr thrown and has
//	   Precompiled headers disabled, with the following text:
//			#include <afxwin.h>
//			#include <afxdllx.h>
#endif

//#ifdef _DEBUG
//#define new DEBUG_NEW
//#endif


static AFX_EXTENSION_MODULE CustomVNAinterfaceDLL = { NULL, NULL };

#ifdef _MANAGED
#pragma managed(push, off)
#endif

extern "C" int APIENTRY
DllMain(HINSTANCE hInstance, DWORD dwReason, LPVOID lpReserved)
{
	// Remove this if you use lpReserved
	UNREFERENCED_PARAMETER(lpReserved);

	if (dwReason == DLL_PROCESS_ATTACH)
	{
		TRACE0("CustomVNAinterface.DLL Initializing!\n");
		
		// Extension DLL one-time initialization
		if (!AfxInitExtensionModule(CustomVNAinterfaceDLL, hInstance))
			return 0;

		// Insert this DLL into the resource chain
		// NOTE: If this Extension DLL is being implicitly linked to by
		//  an MFC Regular DLL (such as an ActiveX Control)
		//  instead of an MFC application, then you will want to
		//  remove this line from DllMain and put it in a separate
		//  function exported from this Extension DLL.  The Regular DLL
		//  that uses this Extension DLL should then explicitly call that
		//  function to initialize this Extension DLL.  Otherwise,
		//  the CDynLinkLibrary object will not be attached to the
		//  Regular DLL's resource chain, and serious problems will
		//  result.

		new CDynLinkLibrary(CustomVNAinterfaceDLL);

	}
	else if (dwReason == DLL_PROCESS_DETACH)
	{
		TRACE0("CustomVNAinterface.DLL Terminating!\n");

		// Terminate the library before destructors are called
		AfxTermExtensionModule(CustomVNAinterfaceDLL);
	}
	return 1;   // ok
}

#ifdef _MANAGED
#pragma managed(pop)
#endif


// *****************************************************************************
// the rest of this file contains the hardware specific functions.
// replace these as needed but keep the functions and their declarations unchanged
// *****************************************************************************
// This set of functions provides access to the VNA via a USB interface
//
// Call the CustomVNAhwInit function on startup and then pass the value it returns
// (which is a pointer to CustomVNAhw) to every function called
// and remember to call CustomVNAhwFree before closing

// *****************************************************************************
//  creator for a CustomVNAhw device
//
// Input parameters:
//		nInstance - currently unused - this is intended for use to specify one of a number of VNA instances
// Return code:
//		NULL - could not create object.
//		!= NULL; Pointer to CustomVNAhw object. This will be passed back to each ot the other functions
//		


__declspec(dllexport) void * _stdcall CustomVNAhwInit(short nInstance)
{
	return (void *)(new CustomVNAhw);
}

// *****************************************************************************
//  destructor for a CustomVNAhw device
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
// Return code:
//		none

__declspec(dllexport) void _stdcall CustomVNAhwFree(void *pHelper)
{
	delete (CustomVNAhw *)pHelper;
}

// *****************************************************************************
// CheckVNAHardware() - see if we think we have a connected working VNA
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
// Return code:
//		true if we appear to have a working VNA interface
//		false if not

__declspec(dllexport) bool _stdcall CheckVNAHardware(void *pHelper)
{
	// just in case we get a null pointer. Error if so.
	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return false;

	if( ((CustomVNAhw *)pHelper)->dVNA->get_State() != 1 )
		return false;
	else
		return true;
}

// *****************************************************************************
// Called when we want to use the VNA but the above function says we can't
// Try dropping the connection and trying again
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		nInstance - currently unused - this is intended for use to specify one of a number of VNA instances
// Return code:
//		true if we appear to have a working VNA interface
//		false if not

__declspec(dllexport) bool _stdcall RescanForCustomVNA(void *pHelper, int nInstance)
{
	if( pHelper  == NULL )
		return false;
	delete ((CustomVNAhw *)pHelper)->dVNA;
	((CustomVNAhw *)pHelper)->dVNA = new VNADevice;
	return CheckVNAHardware(pHelper);
}

// *****************************************************************************
// Get device dll versions
// retrieve the version numbers of the underlying device access dll
// used for version control
// and also to make sure that the software stack appears to be functional
// If you have nothing specific to return, just return 0
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		pCustomVNAinterfaceDLLversion - pointer to version information for this DLL
//		pVnaDllVersion - pointer to version information for underlying driver
//		pUSBCodeVersion - pointer to version information for hardware itself
// Return code:
//		false: cannot initialise interface
//		true: drivers working. DLL and hardare interface code versions are returned to caller

__declspec(dllexport) bool _stdcall GetCustomVNACodeVersions(void *pHelper, int *pCustomVNAinterfaceDLLversion, int *pVnaDllVersion, int *pUSBCodeVersion )
{
	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return false;
	if( !((CustomVNAhw *)pHelper)->dVNA->Init())
		return false;
	int i = ((CustomVNAhw *)pHelper)->dVNA->GetVersions();
	*pVnaDllVersion = (i>>8) & 0xff;
	*pUSBCodeVersion = i & 0xff;
	pCustomVNAinterfaceDLLversion = 0;
	return true;
}

// *****************************************************************************
// Send a command to the VNA.
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		DDS clock frequency in Hz
//		LO and RF DDS settings, (frequency in Hz and phase for each - see below)
//		delay before ADC reads in usec, 
//		ADC Count - number of ADC reads for this DDS setting
//		ADC mode - see below
//		bPauseInOut - flag to signal if should block reads until data is ready
//		bOverlapDDSandADCFlag - flag to say whether DDS can be loaded ready for next step whilst waiting for ADC conversion
//		bAdcOnly - flag that indicates that no DDS setting is to be done, just ADC reading
// Return code:
//		return codes defined in CustomVNAinterfaceDLL.h.
//		return either VNAHW_OK or VNAHW_HWERROR
//
// Frequencies are in Hz
//
// Phase is in units of 1/1000 of a degree if the DLL reported from 
// CustomVNAGetInformation() that it wanted phase in degrees
// but in this example we said to tell us in multiples of 11.25 degrees 
// ( bit set and 5 bits of phase so each unit is 360/(2^5) or 360/32 or 11.25
// so as the AD9851 units of "11.25 degrees" means 90 degrees needs a setting of 8
// means we will get a value of 8 pased and need to shift it left 3 bits
//
//
// Mode is structured as follows. Bits 0..5 are the LTC2440 OSR values, bit 6 if set causes ADC 1 and 2
// to be read simultaneously and bit 7 if set uses ADC2 instead of ADC1 for non simulatneous converts
//
// refer to USB documentation for details on the flags
//
//
// This is one of two main functions used in a scan, the other being to read the VNA reading data
// Unless bAdcOnly is set, the DDS chips should be set to this frequency and an ADC conversion started
// The specified delay before the ADC conversions should be honoured.
// The function does *NOT* wait for the ADC conversion results, it is expected to return as soon as the
// command has been scheduled for execution. In the USB interface, all this means is that it is sent to the USB controller chip.
// and it will be queued waiting in the controller.
// However if the overlap flag is set, then the USB controller will have a peek at the queued up command and to the loading
// of the indicated data whilst waiting for ADC conversion so that it is sitting ready in the DDS chip and only needs an FQUD pulse

__declspec(dllexport) int _stdcall CustomVNACommand(void *pHelper, double dClockDDS, double dLoDDS, double dRfDDS, int LoPhase, int RfPhase, 
							int Delay, int AdcCount, int AdcMode, bool bPauseInOut, bool bOverlapDDSandADCFlag, bool bAdcOnly )
{
	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return VNAHW_HWERROR;

	short retcode = VNAHW_OK;
	__int64 LoDDS, RfDDS;

	LoDDS = (__int64)(dLoDDS / dClockDDS * (double)4294967296);
	RfDDS = (__int64)(dRfDDS / dClockDDS * (double)4294967296);

	VNA_TXBUFFER_CMD CommandData;
	// This is the USB "SET" command
	CommandData.command_code = 0x55;
	// Delay is in usec, so if > 2000 we specify in msec, othwerwise we use units of (usec/8)
	// see the USB interface guide for more details
	CommandData.adc_delay = (unsigned char)(Delay > 2000 ? Delay / 1000 : Delay / 8);
	CommandData.adc_mode = (unsigned char)AdcMode;
	CommandData.adc_reads = (unsigned char)AdcCount;
	CommandData.flags = 0;
	if( ! bAdcOnly )
		CommandData.flags = bCmdVnaSetDdsFlagsDdsSet | (Delay <= 2000 ? bCmdVnaSetDdsFlagsDelayIsUsec : 0);
	if( bPauseInOut )
		CommandData.flags |= bCmdVnaSetDdsFlagsPauseDataIn | bCmdVnaSetDdsFlagsPauseDataOut;
	if( bOverlapDDSandADCFlag )
		CommandData.flags |= bCmdVnaSetDdsFlagsPreloadDDS;

#ifdef PASS_DEGREES
	// the following line would be needed if we got phase as (degrees*1000) 
	CommandData.lo[0] = ((unsigned char)(((LoPhase/11250)&PHASEMASK)) << 3);
#else
	// but in this case we said to give us multiples of 11.25 degree steps so we do this
	// same also true of RF phase below
	CommandData.lo[0] = ((unsigned char)((int(LoPhase)&PHASEMASK)) << 3);
#endif
	CommandData.lo[4] = (unsigned char)LoDDS & 0xff;
	CommandData.lo[3] = (unsigned char)(LoDDS>>8) & 0xff;
	CommandData.lo[2] = (unsigned char)(LoDDS>>16) & 0xff;
	CommandData.lo[1] = (unsigned char)(LoDDS>>24) & 0xff;

	// see the above example of phase managment - do same here
#ifdef PASS_DEGREES
	CommandData.rf[0] = ((unsigned char)(((RfPhase/11250)&PHASEMASK)) << 3);
#else
	CommandData.rf[0] = ((unsigned char)((int(RfPhase)&PHASEMASK)) << 3);
#endif
	CommandData.rf[4] = (unsigned char)RfDDS & 0xff;
	CommandData.rf[3] = (unsigned char)(RfDDS>>8) & 0xff;
	CommandData.rf[2] = (unsigned char)(RfDDS>>16) & 0xff;
	CommandData.rf[1] = (unsigned char)(RfDDS>>24) & 0xff;
	if( !((CustomVNAhw *)pHelper)->dVNA->Write( (union _VNA_TXBUFFER *)(&CommandData), sizeof( CommandData )))
		retcode = VNAHW_HWERROR;
	return retcode;
}

// *****************************************************************************
// Read the VNA. On return, flags will contain the VNA status
// in the flags, a count of the number of ADC reads performed
// and the data read in the array pointed to by *data
// Note that if an over / under range occurs in the ADC conversion, 
// we set the flags status bit. Note we expand this from a byte to an int
//
// The calling function will, if it is expecting data, call read again.
// It is expected that the Read() function will block the thread until data
// is available if a read is in progress to stop wasting CPU time, which is 
// what the USB stack does using the PauseDataIn flag (see above).
// In order to support this, the function calling this routine in these events
// will run as a separate thread so that the GUI is not blocked.
//
// However - if there is no read in progress, DO NOT BLOCK UNTIL DATA IS READY
// because of course data will in that case never be available and it will hang.
// if there is no data then return a count of 0

// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		pflags - pointer to set of flags that indicate result information such as overflow
//		pcount - pointer to returned cound of number of ADC reads
//		pDataReturned - pointer to location where resyultin readings should be stored as signed long values
// Return code:
//			VNAHW_OK = OK, function completed correctly
//			VNAHW_HWERROR = Error returned from hardware device IO call
// note these are defined in vnaio.h

__declspec(dllexport) int _stdcall CustomVNARead(void *pHelper, int *pflags, int *pcount, signed long *pDataReturned)
{
	VNA_RXBUFFER DataReceived;
	unsigned char *pDataTemp = &DataReceived.data[0];
	int i;
	int nRetcode = VNAHW_OK;

	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return VNAHW_HWERROR;
	if( !((CustomVNAhw *)pHelper)->dVNA->Read( &DataReceived ))
		nRetcode = VNAHW_HWERROR;
	*pflags = DataReceived.return_status;
	*pcount = 0;
	if( nRetcode == VNAHW_OK && DataReceived.ADC_reads_done > 0 )
	{
		for( i=0; i< DataReceived.ADC_reads_done; i++)
		{
			pDataReturned[i] = ((*pDataTemp) <<24)+(*(pDataTemp+1)<<16)+(*(pDataTemp+2)<<8)+(*(pDataTemp+3));
			if( (pDataReturned[i] & 0x30000000 ) == 0 )
				// overflow
				*pflags |= ADC_OVERRANGE;
			else if( (pDataReturned[i] & 0x30000000 ) == 0x30000000 )
				// overflow
				*pflags |= ADC_OVERRANGE;
			else
			{
				unsigned long x = pDataReturned[i] & 0x1fffffff;
				if( (pDataReturned[i] & 0x20000000 ) == 0 )
					x |= 0xe0000000;
				pDataReturned[i] = x;
			}
			pDataTemp += 4;
		}
		*pcount = DataReceived.ADC_reads_done;
	}
	return nRetcode;
}

// *****************************************************************************
// This is called once we know that hardware is available.
// Keep reading from ADC until any pending data is flushed
// then check the status of the flags.
// Any flag set at this point reflects an error
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		nInstance - currently unused - this is intended for use to specify one of a number of VNA instances
// Return code:
//		set of flags that indicate status
// Current errors are
// ADC_OVERRANGE
// ADC_READ_TIMEOUT
// VNA_NO_POWER
// ADC_START_FLAG
// ADC_NOT_RESPONDING (map DeviceIOControl error here also)

__declspec(dllexport) int _stdcall CheckCustomVNAReady(void *pHelper, int nInstance)
{
	int flags = ADC_DATA_READY;
	int count;
	int response;
	long data[64];

	response = CustomVNARead( pHelper, &flags, &count, &data[0] );
	if( response != VNAHW_OK )
		RescanForCustomVNA(pHelper, nInstance);
	response = CustomVNARead( pHelper, &flags, &count, &data[0] );
	response = CustomVNARead( pHelper, &flags, &count, &data[0] );
	// keep reading until pending read data is flushed
	while( flags & bVnaStatusAdcDataReadyFlag )
		response = CustomVNARead(pHelper,  &flags, &count, &data[0] );
	// check for over / under range
	if( response == VNAHW_OVERRANGE )
		flags |= ADC_OVERRANGE;
	if( response == VNAHW_HWERROR )
		flags |= ADC_NOT_RESPONDING;
	return flags;	
}

// *****************************************************************************
// power down the VNA (well, the DDS chips anyway)
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
// Return code:
//		VNAHW_OK or VNAHW_ERROR

__declspec(dllexport) int _stdcall PowerDownCustomVNA(void *pHelper)
{
	int retcode = VNAHW_OK;

	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return VNAHW_HWERROR;

	VNA_TXBUFFER_CMD CommandData;
	// This is the USB "SET" command
	CommandData.command_code = 0x55;
	// Delay is in usec, so if > 2000 we specify in msec, othwerwise we use units of (usec/8)
	// see the USB interface guide for more details
	CommandData.adc_delay = 0;
	CommandData.adc_mode = 0;
	CommandData.adc_reads = 0;
	CommandData.flags =  bCmdVnaSetDdsFlagsDdsSet;

	CommandData.lo[0] = 4;
	CommandData.lo[4] = 0;
	CommandData.lo[3] = 0;
	CommandData.lo[2] = 0;
	CommandData.lo[1] = 0;

	CommandData.rf[0] = 4;
	CommandData.rf[4] = 0;
	CommandData.rf[3] = 0;
	CommandData.rf[2] = 0;
	CommandData.rf[1] = 0;
	if( !((CustomVNAhw *)pHelper)->dVNA->Write( (union _VNA_TXBUFFER *)(&CommandData), sizeof( CommandData )))
		retcode = VNAHW_HWERROR;
	return retcode;
}


// *****************************************************************************
// Reset the DDS chips.
// Send a reset with a set command with PauseDataOut
// The freuency used for the reset is arbitrary
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
// Return code:
//		VNAHW_OK or VNAHW_ERROR

__declspec(dllexport) int _stdcall ResetCustomVNA(void *pHelper)
{
	int nRetcode = VNAHW_OK;
	VNA_TXBUFFER_CMD CommandData;

	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return VNAHW_HWERROR;

	CommandData.command_code = 0x55;
	CommandData.adc_delay = 0;
	CommandData.adc_mode = 0;
	CommandData.adc_reads = 0;
	CommandData.flags = bCmdVnaSetDdsFlagsReset + bCmdVnaSetDdsFlagsPauseDataOut;
	CommandData.lo[0] = 0;
	CommandData.lo[4] = 0;
	CommandData.lo[3] = 0;
	CommandData.lo[2] = 0;
	CommandData.lo[1] = 0;

	CommandData.rf[0] = 0;
	CommandData.rf[4] = 0;
	CommandData.rf[3] = 0;
	CommandData.rf[2] = 0;
	CommandData.rf[1] = 0;
	if( !((CustomVNAhw *)pHelper)->dVNA->Write( (union _VNA_TXBUFFER *)(&CommandData), sizeof( CommandData )))
		nRetcode = VNAHW_HWERROR;
	return nRetcode;
}


// *****************************************************************************
// initiate a sweep function
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		DDS clock frequency in Hz
//		LO and RF DDS settings, (frequency in Hz and phase in either milliDegrees or 
//			multiples of phase step for each)
//		LO and RF DDS step settings, (frequency in Hz and phase ias above for each)
//		the number of steps to make
//		the delay between steps in usec
//		the delay at the start of th sweep before stepping
//		a flag that indicates whether to pulse the SW1 line at sweep start
//		a flag to say if DDS chipe should be powered down briefly between sweepa
// Return code:
//		VNAHW_OK or VNAHW_ERROR
//
// Phase is in units of 1/1000 of a degree
// so the AD9851 units of "11.25 degrees" means 90 degrees needs a setting of 8
// hence divide value given by 11250 to get dds setting

__declspec(dllexport) int _stdcall CustomVNASweep(void *pHelper, double dClockDDS, double dLoDDS, double dRfDDS, int LoPhase, int RfPhase,
				 double dLoStepDDS, double dRfStepDDS, int LoStepPhase, int RfStepPhase, 
				 int nSteps, int nStepDelay, int nInitDelay, bool SetSW1, bool bPowerDownBetweenSweeps)
{
	int retcode =  VNAHW_OK;

	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return VNAHW_HWERROR;

	__int64 LoDDS, RfDDS;
	__int64 LoStepDDS, RfStepDDS;

	LoDDS = (__int64)(dLoDDS / dClockDDS * (double)4294967296);
	RfDDS = (__int64)(dRfDDS / dClockDDS * (double)4294967296);
	LoStepDDS = (__int64)(dLoStepDDS / dClockDDS * (double)4294967296);
	RfStepDDS = (__int64)(dRfStepDDS / dClockDDS * (double)4294967296);

	VNA_TXBUFFER_SWEEP SweepData;
	SweepData.command_code = 0xAA;
	SweepData.flags = 0;

#ifdef PASS_DEGREES
	// the following line would be needed if we got phase as (degrees*1000) 
	SweepData.lo[0] = (unsigned char)(((LoPhase/11250)&PHASEMASK)) << 3;
#else
	// but in this case we said to give us multiples of 11.25 degree steps so we do this
	// same also true of RF phase below (and also the step phase value)
	SweepData.lo[0] = ((unsigned char)((int(LoPhase)&PHASEMASK)) << 3);
#endif
	SweepData.lo[4] = (unsigned char)LoDDS & 0xff;
	SweepData.lo[3] = (unsigned char)(LoDDS>>8) & 0xff;
	SweepData.lo[2] = (unsigned char)(LoDDS>>16) & 0xff;
	SweepData.lo[1] = (unsigned char)(LoDDS>>24) & 0xff;

#ifdef PASS_DEGREES
	// see above note on phase
	SweepData.rf[0] = (unsigned char)(((RfPhase/11250)&PHASEMASK)) << 3;
#else
	SweepData.rf[0] = (unsigned char)((int(RfPhase)&PHASEMASK)) << 3;
#endif
	SweepData.rf[4] = (unsigned char)RfDDS & 0xff;
	SweepData.rf[3] = (unsigned char)(RfDDS>>8) & 0xff;
	SweepData.rf[2] = (unsigned char)(RfDDS>>16) & 0xff;
	SweepData.rf[1] = (unsigned char)(RfDDS>>24) & 0xff;

#ifdef PASS_DEGREES
	// see above note on phase
	SweepData.lostep[0] = (unsigned char)(((LoStepPhase/11250)&PHASEMASK)) << 3;
#else
	SweepData.lostep[0] = (unsigned char)((int(LoStepPhase)&PHASEMASK)) << 3;
#endif
	SweepData.lostep[4] = (unsigned char)LoStepDDS & 0xff;
	SweepData.lostep[3] = (unsigned char)(LoStepDDS>>8) & 0xff;
	SweepData.lostep[2] = (unsigned char)(LoStepDDS>>16) & 0xff;
	SweepData.lostep[1] = (unsigned char)(LoStepDDS>>24) & 0xff;

#ifdef PASS_DEGREES
	// see above note on phase
	SweepData.rfstep[0] = (unsigned char)(((RfStepPhase/11250)&PHASEMASK)) << 3;
#else
	SweepData.rfstep[0] = (unsigned char)((int(RfStepPhase)&PHASEMASK)) << 3;
#endif
	SweepData.rfstep[4] = (unsigned char)RfStepDDS & 0xff;
	SweepData.rfstep[3] = (unsigned char)(RfStepDDS>>8) & 0xff;
	SweepData.rfstep[2] = (unsigned char)(RfStepDDS>>16) & 0xff;
	SweepData.rfstep[1] = (unsigned char)(RfStepDDS>>24) & 0xff;

	SweepData.init_delay = nInitDelay > 2000 ? nInitDelay / 1000 : nInitDelay / 8;
	SweepData.step_delay = nInitDelay > 2000 ? nStepDelay / 1000 : nStepDelay / 8;

	SweepData.steps[3] = (unsigned char)(nSteps)&0xff;
	SweepData.steps[2] = (unsigned char)(nSteps>>8)&0xff;
	SweepData.steps[1] = (unsigned char)(nSteps>>16)&0xff;
	SweepData.steps[0] = (unsigned char)(nSteps>>24)&0xff;

	if( SetSW1 )
		SweepData.flags |= bCmdVnaSweepPulseSW1;
	if( bPowerDownBetweenSweeps )
		SweepData.flags |= bCmdVnaSweepPowerDownbetweenSweeps;
	if( nStepDelay <= 2000 )
		SweepData.flags |= bCmdVnaSweepStepDelayIsUsec;
	if( nInitDelay <= 2000 )
		SweepData.flags |= bCmdVnaSweepInitDelayIsUsec;

	if( !((CustomVNAhw *)pHelper)->dVNA->Write( (union _VNA_TXBUFFER *)(&SweepData), sizeof( SweepData )))
		retcode = VNAHW_HWERROR;
	return retcode;

}


// *****************************************************************************
// This interface is called by the main document to set the switch output lines
// nSetting is a set of bit flags; bit 0 - switch 0, bit 1 = switch 1.
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		the desired switch setting
// Return code:
//		VNAHW_OK or VNAHW_ERROR

__declspec(dllexport) int _stdcall SetCustomVNASwitchLines(void *pHelper, int nSetting )
{
	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return VNAHW_HWERROR;

	_VNA_TXBUFFER_RAW RawData;
	// This is the USB "Config" command
	RawData.command_code = 0x5A;
	RawData.flags = CmdVnaRawDataFlagsSetSwitch;
	RawData.atten = RawData.portA = RawData.portB = RawData.portD = 0;
	RawData.swset = nSetting;
	if( ((CustomVNAhw *)pHelper)->dVNA->Write( (union _VNA_TXBUFFER *)(&RawData), sizeof( RawData )))
		return VNAHW_OK;
	else
		return VNAHW_HWERROR;
}

// *****************************************************************************
// This interface is called by the main document to set the attenuator output lines
// nSetting takes a value 0..7. It is not bounds checked (there is no need)
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		the desired switch setting
// Return code:
//		VNAHW_OK or VNAHW_ERROR

__declspec(dllexport) int _stdcall SetCustomVNAAttenuatorLines(void *pHelper, int nSetting )
{
	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return VNAHW_HWERROR;

	_VNA_TXBUFFER_RAW RawData;
	// This is the USB "Config" command
	RawData.command_code = 0x5A;
	RawData.flags = CmdVnaRawDataFlagsSetAtten;
	RawData.swset = RawData.portA = RawData.portB = RawData.portD = 0;
	RawData.atten = nSetting;
	if( ((CustomVNAhw *)pHelper)->dVNA->Write( (union _VNA_TXBUFFER *)(&RawData), sizeof( RawData )))
		return VNAHW_OK;
	else
		return VNAHW_HWERROR;
}

// *****************************************************************************
// This interface is called by the IO debugging dialog to directly set ports A and B
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		the desired hardware port settings for 2 bytewide ports
// Return code:
//		VNAHW_OK or VNAHW_ERROR

__declspec(dllexport) int _stdcall SetCustomVNAPortABLines(void *pHelper, int nSettingA, int nSettingB )
{
	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return VNAHW_HWERROR;

	_VNA_TXBUFFER_RAW RawData;
	// This is the USB "Config" command
	RawData.command_code = 0x5A;
	RawData.flags = CmdVnaRawDataFlagsWriteA | CmdVnaRawDataFlagsWriteB;
	RawData.atten = RawData.portD = RawData.swset = 0;
	RawData.portA = nSettingA;
	RawData.portB = nSettingB;
	
	if( ((CustomVNAhw *)pHelper)->dVNA->Write( (union _VNA_TXBUFFER *)(&RawData), sizeof( RawData )))
		return VNAHW_OK;
	else
		return VNAHW_HWERROR;
}

// *****************************************************************************
// This interface is called by the IO debugging dialog to directly read ports A and B
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		pointers to the location for the current hardware port settings for 2 bytewide ports
// Return code:
//		VNAHW_OK or VNAHW_ERROR

__declspec(dllexport) int _stdcall CustomVNARawRead(void *pHelper, int *dataA, int *dataB)
{
	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return VNAHW_HWERROR;

	VNA_RXBUFFER DataReceived;
	int nRetcode = VNAHW_OK;

	if( !((CustomVNAhw *)pHelper)->dVNA->Read( &DataReceived ))
		nRetcode = VNAHW_HWERROR;
	*dataA = DataReceived.ioa;
	*dataB = DataReceived.iob;
	return nRetcode;
}

// *****************************************************************************
// This interface is called to determine how to communicate with the hardware
//
// Input parameters:
//		pHelper - Pointer to CustomVNAhw object.
//		Version number for the structure that holds the reaults
//		Size of the struture for the results (just being cautious)
//		what information is requested
//		pointers to the location for the return data
// Return code:
//		true - understood request and gave results
//		false - don't know what you were asking for or structure unkn own or size wrong
//
// At the moment the following only are known:
// what = 1 - give me basic hardware details
//		structure version 1 - array of unsigned chars as follows
//		structure size 4
//		pResult[0] - number of bytes returned including this one
//		pResult[1] - number of switch lines
//		pResult[2] - number of attenuator lines
//		pResult[3] - number of bits in DDSes
//		pResult[4] - flags that indicate functions supported
//			bit 0 set - VNA
//			bit 1 set - Spectrum analyser
//			bit 2 set - VVM
//			bit 3 set - SIgnal Generator function
//		pResult[4] - flags that indicate how to pass data
//			bit 0 clear - give me phase in degrees
//				  set-give me phase as a fraction as follows
//				  pResults[6] gives the number of bits of phase
//				  for example if it takes the value 8 then the
//				  program expects phase as multiples of 360/256 degrees
//				  hence if the caller wants 90 degrees it will
//				  send a phase value of 64.000 if the bit is set and 90.000 if not
//			bit 1 clear - work as described above with PauseInOut flags and blocking
//				  set - don't use blocking model for setting DS and reading (NOT YET SUPPORTED)
//		pResult[6] - number of bits of phase. Even if you expect phase in degrees set this
//				  because it is needed to decide on supported filter modes
//				  for 32 steps of 11.25 degrees set this to 5, for 256 steps of 1.40625 set it to 8
//				  if you only support 0 & 90, set it to 1 etc.
//		pResult[7] - number of bits in DDS tuning words

__declspec(dllexport) int _stdcall CustomVNAGetInformation(void *pHelper, int StructVersion, int StructSize, int what, void *pResult)
{
	if( pHelper == NULL || ((CustomVNAhw *)pHelper)->dVNA == NULL )
		return false;
	char *pChar = (char *)pResult;
	switch( what )
	{
	case 1: // basic hardware details
		if( StructVersion == 1 && StructSize >= 7 )
		{
			pChar[0] = 7;		// 6 bytes follow this	
			pChar[1] = 2;		// 2 switch lines
			pChar[2] = 3;		// 3 attenuator lines
			pChar[3] = 32;		// DDS frequency settings are 32 bits long
			pChar[4] = 0x0D;	// signal generator, VVM and VNA modes
#ifdef PASS_DEGREES
			pChar[5] = 0;		// give me phase as degrees * 1000
#else
			pChar[5] = 1;		// give me phase as multiples of value that follows
#endif
			pChar[6] = PHASE_BITS;		// 2^5 or 32 phase steps so a phase value of 4 means 45 degrees
			return true;
		}
	}
	return false;
}

