#ifndef _VNAIO_
#define _VNAIO_

// plagiarised from the TAPR VNA design.
// as a consequence, the rights of Thomas C. McDermott, N5EG are acknowledged.
// Here is the licence text from his code. The only part used in this way
// is the basic structure of the VNA device and helper.
// All else is different 
// All the other code is Copyright (C) Dave Roberts G8KBB 2004
//
// ----------------- Extract from USB_EZ_interface.cpp -----------------
//    Copyright 2004, Thomas C. McDermott, N5EG
//    This file is part of VNAR - the Vector Network Analyzer program.
//
//    VNAR is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    VNAR is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with VNAR, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// ------------------------End Extract ----------------------------------

// includes extensions for Dec 06 driver
// includes extensions for 9912 VNA

#include "stdafx.h"

extern "C"
{
	// Declare USB constants and structures
//	#include "C:\Cypress\USB\Drivers\ezusbdrv\ezusbsys.h"  // Ezusb IOCTL codes
	#include "ezusbsys.h"
}

#define USB_STRING   char[256]

//-----------------------------------------------------------------------------
// This is the structure used for messages returned to the caller
// from the USB chip.

typedef struct _VNA_RXBUFFER {
	unsigned char last_command; // command type last received
	unsigned char return_status; // see below for status flag defintions
	unsigned char ioa;  // FX2 PortA data
	unsigned char iob;  // FX2 PortB data
	unsigned char ADC_reads_done; // Number of ADC reads performed
	unsigned char data[250]; // VARIABLE nuumber of ADC reads performed
} VNA_RXBUFFER;

// bit definitions in VNA_RXBUFFER.return_status
//
#define bVnaStatusAdcTimeoutFlag 0x80 // If set, FX2 timed out ADC read operation
#define bVnaStatusNoVnaPowerFlag 0x40 // If set, portB bit 7 is low so no VNA power
#define bVnaStatusAdcDataReadyFlag 0x20 // If set, the data[] part conatins ADC data
#define bVnaStatusAdcConvPendFlag 0x10 // Internal flag, if high FX2 is waiting to start ADC read operations
#define bVnaStatusAdcNotRespondingFlag 0x08 // Internal flag, diagnostic info - see documentation
#define bVnaStatusI2CLastOpFailed	0x04			   // I2C read/write status from last call (9912 VNA only)
#define bVnaStatusI2CDataReady	0x02			   // I2C Data is Available (9912 VNA only)


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// There are several types of message that we may send to the VNA. This is managed as a union.
// The first one is a raw port write to FX2 ports A and/or B.
typedef struct _VNA_TXBUFFER_RAW {
	unsigned char command_code;  // set to 0x5A to signal raw write
	unsigned char flags;   // see below for definitions  
	unsigned char portA;   // value to write to port A if flag set
	unsigned char portB;   // value to write to port B if flag set
	unsigned char atten;   // value to write to attenuator bits if flag set
	unsigned char portD;   // value to write to port B if flag set
	unsigned char swset;   // value to write to switch bits if flag set
	unsigned char portC;   // value to write to port C if flag set (9912 VNA only)
} VNA_TXBUFFER_RAW;

//
// bit definitions for Flags for raw command
//
#define CmdVnaRawDataFlagsWriteA 0x80              //if set, write specified value to port A
#define CmdVnaRawDataFlagsWriteB 0x40              //if set, write specified value to port B
#define CmdVnaRawDataFlagsSetAtten 0x20            //if set, wrtite specified value to attenuator bits
#define CmdVnaRawDataFlagsWriteD 0x10              //if set, write specifiied value to port D
#define CmdVnaRawDataFlagsSetSwitch 0x08           //if set, write specified value to switch port
#define CmdVnaRawDataFlagsWriteC 0x04              //if set, write specifiied value to port C (9912 VNA only)


//-----------------------------------------------------------------------------
// This is the second message type we might send. A high level command to the FX2 CPU
// The flags tell the FX2 what to do with the DDS (reset them / write to them )
// After this, it will delay at least adc_del msec before doing ADC reads
// It then performs a series of back to back ADC reads before returning the result
// of all ADC reads in a message as structured above (VNA_RXBUFFER).
typedef struct _VNA_TXBUFFER_CMD {
	unsigned char command_code;  // command code 0x55 for Set command
	unsigned char flags;   // see belo for definitions
	unsigned char adc_delay;  // min delay period (0-255 msec) before ADC reads
	unsigned char adc_reads;  // number of ADC reads to perform
	unsigned char adc_mode;   // adc selection & OSR bits
	unsigned char lo[5];   // LO data to write to DDS
	unsigned char rf[5];   // RF data to write to DDS
	unsigned char lo2[5];   // optional second set of LO data
	unsigned char rf2[5];   // optional second set of RF data
} VNA_TXBUFFER_CMD;

// this is the alternative to the above for the 9912 VNA

typedef struct _VNA_TXBUFFER_CMD_9912_LONG {
	unsigned char command_code;  // command code 0x56 for Set command 
	unsigned char flags;   // see belo for definitions
	unsigned char adc_delay;  // min delay period (0-255 msec) before ADC reads
	unsigned char adc_reads;  // number of ADC reads to perform
	unsigned char adc_mode;   // adc selection & OSR bits
	unsigned char lo[8];   // LO data to write to DDS
	unsigned char rf[8];   // RF data to write to DDS
	unsigned char lo2[8];   // optional second set of LO data
	unsigned char rf2[8];   // optional second set of RF data
} VNA_TXBUFFER_CMD_9912_LONG;

typedef struct _VNA_TXBUFFER_CMD_9912 {
	unsigned char command_code;  // command code 0x56 for Set command 
	unsigned char flags;   // see belo for definitions
	unsigned char adc_delay;  // min delay period (0-255 msec) before ADC reads
	unsigned char adc_reads;  // number of ADC reads to perform
	unsigned char adc_mode;   // adc selection & OSR bits
	unsigned char lo[8];   // LO data to write to DDS
	unsigned char rf[8];   // RF data to write to DDS
} VNA_TXBUFFER_CMD_9912;

//
// bit definitions for flags for set command
//
#define bCmdVnaSetDdsFlagsReset           0x80  //if set, reset DDS
#define bCmdVnaSetDdsFlagsDdsSet          0x40  //if set, configure both DDS
#define bCmdVnaSetDdsFlagsDelayIsUsec     0x20  //if set, delay AdcDel is in usec not millisec
#define bCmdVnaSetDdsFlagsPauseDataIn     0x10  //if set, stop returning IN data until ADC conversions done
#define bCmdVnaSetDdsFlagsPauseDataOut    0x08  //if set, do not process incoming messages till ADC done
#define bCmdVnaSetDdsFlagsSingleFunction  0x04  //if set, bit6 sets DDS but no FQ_UD, if clear bit 6 pulses FQ_UD only
#define bCmdVnaSetDdsFlagsDoubleConvert   0x02  //if set, use second set of DDS data and do two readings of DDS
#define bCmdVnaSetDdsFlagsPreloadDDS      0x01  // if set preload DDS during ADC conversion
//
// bit defintions for adc_mode for set command
//
#define CmdVnaSetDdsAdcModeDet2          0x80   //if set use second detector
#define CmdVnaSetDdsAdcModeSimultaneous  0x40   //if set use both detectors
// bit 5 unused
// bits 4..0 are LTC2440 OSR4..0

// bit definitions for ADCcount
// if bit 7 is set, force switch lines from bits 6,5 when DDS is set or when starting conversion
#define CmdVnaSetDdsAutoSwitch			0x80
#define CmdVnaSetDdsAutoSwitch1			0x40
#define CmdVnaSetDdsAutoSwitch0			0x20
// bottom 5 bits are the ADC count itself
#define ADC_COUNT_MASK 		(0x1f)

//-----------------------------------------------------------------------------
// This is the third message type we might send. 
// allows setting an override for the ADc mode and imposing a minimum delay beore ADC conversions
typedef struct _VNA_TXBUFFER_CONFIGURE {
	unsigned char command_code;  // command code 0xA5 for configure command
	unsigned char flags;   // see below for definitions
	unsigned char mode;   // ADC mode override setting
	unsigned char min_delay;  // minimum ADC conversion time setting
} VNA_TXBUFFER_CONFIGURE;

//
// bit definitions for flags for configure command
//
#define bCmdVnaConfModeOverride           0x80  //if set, override ADC mode
#define bCmdVnaConfADCMinDelay            0x40  //if set, enforce min ADC delay


//-----------------------------------------------------------------------------
// This is the fourth message type we might send. 
// allows us to program the VNA to do an accurate time sweep of DDS
typedef struct _VNA_TXBUFFER_SWEEP {
	unsigned char command_code;  // command code 0xAA for sweep command
	unsigned char flags;   // see below for definitions
	unsigned char step_delay;  // delay between steps
	unsigned char init_delay;  // delay before next sweep
	unsigned char steps[4];                 // number of steps in sweep
	unsigned char lo[5];                    // starting local osc DDS setting
	unsigned char rf[5];                    // starting RF DDS setting
	unsigned char lostep[5];                // step increment for LO DDS
	unsigned char rfstep[5];                // step increment for RF DDS
} VNA_TXBUFFER_SWEEP;

// This is the 9912 version of the fourth message type we might send. 
// allows us to program the VNA to do an accurate time sweep of DDS
// this is the 9912 version (NOT YET IMPLEMENTED)

typedef struct _VNA_TXBUFFER_SWEEP_9912 {
	unsigned char command_code;  // command code AB for sweep command
	unsigned char flags;   // see below for definitions
	unsigned char step_delay;  // delay between steps
	unsigned char init_delay;  // delay before next sweep
	unsigned char steps[4];                 // number of steps in sweep
	unsigned char lo[8];                    // starting local osc DDS setting
	unsigned char rf[8];                    // starting RF DDS setting
	unsigned char lostep[8];                // step increment for LO DDS
	unsigned char rfstep[8];                // step increment for RF DDS
} VNA_TXBUFFER_SWEEP_9912;

//
// bit definitions for flags for sweep command
//
#define bCmdVnaSweepPulseSW1           0x80     //if set, pulse the SW1 line between sweeps
#define bCmdVnaSweepStepDelayIsUsec    0x40     //if set, step delay is usec * 8 setting
#define bCmdVnaSweepInitDelayIsUsec    0x20     // if set initial sweep delay is usec * 8 setting
#define bCmdVnaSweepPowerDownbetweenSweeps 0x10 // if set power down the DDSes between sweeps to switch outputs off

//-----------------------------------------------------------------------------
// for the 9912 VNA we also have I2C read, I2C Write and DDS write
// Write to I2C (9912 VNA only)
typedef struct _VNA_TXBUFFER_I2CWRITE {
	unsigned char command_code;  // command code 0xB6 for I2C Write command
	unsigned char flags;   // see below for definitions
	unsigned char count;  // count of data bytes
	unsigned char address;  // sub address
	unsigned char data[56];  // data to write ( avoid 64 byte (could be 60 but being cautious....)
} VNA_TXBUFFER_I2CWRITE;
// flag bit definitions for I2C Write
#define bCmdVnaI2CWriteFlag400kHz   0x80		   // set to run bus at 400 kHz instead of 100 kHz

//-----------------------------------------------------------------------------
// Read from  I2C (9912 VNA only)
typedef struct _VNA_TXBUFFER_I2CREAD {
	unsigned char command_code;  // command code 0xB9 for sweep command
	unsigned char flags;   // see below for definitions
	unsigned char readcount;  // count of data bytes to read
	unsigned char writecount;  // count of data bytes to read
	unsigned char address;  // sub address
	unsigned char writedata[56];  // data to write ( avoid 64 byte (could be 60 but being cautious....)
} VNA_TXBUFFER_I2CREAD;
// flag bit definitions for I2C Read
#define CmdVnaI2CReadFlag400kHz 0x80		   // set to run bus at 400 kHz instead of 100 kHz

//-----------------------------------------------------------------------------
// Write to DDS (9912 VNA only)
// The data is LO data followed by RF data. Both fields must be present and of length (count)
// even when only one DDS is being written.
typedef struct _VNA_TXBUFFER_9912WRITE {
	unsigned char command_code;  // command code 0x6B for DDS write command
	unsigned char flags;   // see below for definitions
	unsigned char count;  // count of data bytes to read
	unsigned char address1;  // start address ms byte
	unsigned char address2;  // start address ls byte
	unsigned char writedata[56];  // data to write ( avoid 64 byte (could be 60 but being cautious....)
} VNA_TXBUFFER_9912WRITE;
// flag bit definitions for DDS Write
#define CmdVnaDdsWriteFlagReset   0x80
#define CmdVnaDdsWriteFlagUpdate  0x40
#define CmdVnaDdsWriteFlagWriteRf 0x20
#define CmdVnaDdsWriteFlagWriteLo 0x10


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// And this is the union for the messages
typedef union _VNA_TXBUFFER {
	VNA_TXBUFFER_RAW raw;
	VNA_TXBUFFER_CMD cmd;
	VNA_TXBUFFER_CONFIGURE config;
	VNA_TXBUFFER_SWEEP sweep;
	VNA_TXBUFFER_SWEEP_9912 sweep9912;
	VNA_TXBUFFER_CMD_9912 cmd9912;
	VNA_TXBUFFER_CMD_9912_LONG cmd9912long;
	VNA_TXBUFFER_I2CWRITE i2cwrite;
} VNA_TXBUFFER;


// ****************************************************************
// Here is the main interface to the VNA - the VNADevice class.
// the IO32 and vnawrapper functions use this. Use it if you can
// otherwise use vnawrapper, and if all else fails use IO32 but it is slow

class VNADevice
{
public:
	bool Result;     // DeviceIoControl result
	int state;      // -1=no device +1=device OK
	class Helper * d;    // holds the USB device state
	void GetHandle(void);
	void ReleaseHandle(void);
	bool ToggleReset(bool hold);
public:
	__declspec(dllexport) _stdcall VNADevice();   // Constructor: open device, set state
	__declspec(dllexport) _stdcall ~VNADevice();  // Destructor: release __nogc objects and structs
	__declspec(dllexport) bool _stdcall  Init(void); // Build descriptors, get pipes
	__declspec(dllexport) int _stdcall get_State();  // -1 = no device  +1 = device OK
	__declspec(dllexport)  bool _stdcall Start();  // Release reset of the 8051 processor on VNA
	__declspec(dllexport)  bool _stdcall Stop();  // Halt the 8051 processor on VNA
	__declspec(dllexport) int _stdcall get_Instance(); // get instance of VNA (0..9)
	__declspec(dllexport) bool _stdcall set_Instance(int instance); // set instance (0..9)
	__declspec(dllexport)  int _stdcall get_BytesReturned(); // tell me how many bytes were read last time
	__declspec(dllexport)  bool _stdcall Read(VNA_RXBUFFER * readbuf);  // read the VNA
	__declspec(dllexport)  bool _stdcall Write(VNA_TXBUFFER * writebuf, int message_size); // write to the VNA
	__declspec(dllexport) int _stdcall VNADevice::GetVersions();
	__declspec(dllexport) int _stdcall VNADevice::GetDeviceId();
};
// emulation of parallel port driver interface
//bool __declspec(dllexport) _stdcall Out32(short PortAddress, short data);
//short __declspec(dllexport) _stdcall Inp32(short PortAddress);
// simple VNA usb interface
bool __declspec(dllexport) _stdcall vnawrite(void *message, short bytecount);
bool __declspec(dllexport) _stdcall vnaread(void *message, short *bytecount);

#endif
